/*
 * Copyright 2004-2008 H2 Group. Multiple-Licensed under the H2 License, 
 * Version 1.0, and under the Eclipse Public License, Version 1.0
 * (http://h2database.com/html/license.html).
 * Initial Developer: H2 Group
 */
package tim.sql.h2parser;

import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.sql.SQLException;



/**
 * Messages used in the database engine. Use the PropertiesToUTF8 tool to
 * translate properties files to UTF-8 and back. If the word 'SQL' appears then
 * the whole SQL statement must be a parameter, otherwise this may be added: ';
 * SQL statement: ' + sql
 */
public class Message {

   /**
     * Constructs a syntax error SQL exception.
     * 
     * @param sql the SQL statement
     * @param index the position of the error in the SQL statement
     * @return the SQLException object
     */
    public static SQLException getSyntaxError(String sql, int index) {
        sql = StringUtils.addAsterisk(sql, index);
        return new SQLException("SYNTAX_ERROR_1\n"+ sql);
    }
    
    public static SQLException getSyntaxError(String sql, int index, String expected) {
        sql = StringUtils.addAsterisk(sql, index);
        return new SQLException("SYNTAX_ERROR_2\n"+ sql + "\n-- expected: " + expected);
    }
    
    public static SQLException getUnsupportedError(String sql, int index) {
        sql = StringUtils.addAsterisk(sql, index);
        return new SQLException("UNSUPPORTED_ERROR_1\n"+ sql);
    }
    
    public static SQLException getUnsupportedError(String sql, int index, String expected) {
        sql = StringUtils.addAsterisk(sql, index);
        return new SQLException("UNSUPPORTED_ERROR_2\n"+ sql + "\n-- expected: " + expected);
    }
    
    public static SQLException getSQLException(String errorCode) {
        return new SQLException(errorCode);
    }

    public static SQLException getSQLException(String errorCode, String message) {
        return new SQLException(errorCode+"\n"+ message);
    }
    
    public static SQLException getSQLException(String errorCode, String message, Throwable cause) {
        return new SQLException(errorCode+"\n"+ message, cause);
    }
    
	public static Exception getSQLException(String string, String[] strings,
			NumberFormatException e) {
		// TODO Auto-generated method stub
		return null;
	}

    public static SQLException convert(Throwable e) {
//        if (e instanceof InternalException) {
//            e = ((InternalException) e).getOriginalCause();
//        }
        if (e instanceof SQLException) {
            return (SQLException) e;
        } else if (e instanceof OutOfMemoryError) {
            return getSQLException("OUT_OF_MEMORY", null, e);
        } else if (e instanceof InvocationTargetException) {
            InvocationTargetException te = (InvocationTargetException) e;
            Throwable t = te.getTargetException();
            if (t instanceof SQLException) {
                return (SQLException) t;
            }
            return getSQLException("EXCEPTION_IN_FUNCTION", null, e);
        } else if (e instanceof IOException) {
            return getSQLException("IO_EXCEPTION_1", e.toString(), e);
        }
        return getSQLException("GENERAL_ERROR_1", e.toString(), e);
    }


    /**
     * Gets an internal error.
     * 
     * @param s the message
     * @return the RuntimeException object
     */
    public static RuntimeException getInternalError(String s) {
        RuntimeException e = new RuntimeException(s);
        return e;
    }

    /**
     * Gets an internal error.
     * 
     * @param s the message
     * @param e the root cause
     * @return the error object
     */
    public static Error getInternalError(String s, Exception e) {
        return new Error(s,e);
    }
}
