/*
 * Copyright 2004-2008 H2 Group. Multiple-Licensed under the H2 License, 
 * Version 1.0, and under the Eclipse Public License, Version 1.0
 * (http://h2database.com/html/license.html).
 * Initial Developer: H2 Group
 */
package tim.sql.h2parser;

import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;


import tim.sql.h2parser.dml.Query;
import tim.sql.h2parser.dml.Select;
import tim.sql.h2parser.dml.SelectOrderBy;
import tim.sql.h2parser.expr.Aggregate;
import tim.sql.h2parser.expr.Alias;
import tim.sql.h2parser.expr.CompareLike;
import tim.sql.h2parser.expr.Comparison;
import tim.sql.h2parser.expr.ConditionAndOr;
import tim.sql.h2parser.expr.ConditionExists;
import tim.sql.h2parser.expr.ConditionNot;
import tim.sql.h2parser.expr.Expression;
import tim.sql.h2parser.expr.ExpressionColumn;
import tim.sql.h2parser.expr.Function;
import tim.sql.h2parser.expr.Operation;
import tim.sql.h2parser.expr.Parameter;
import tim.sql.h2parser.expr.ValueExpression;
import tim.sql.h2parser.expr.Wildcard;
import tim.sql.h2parser.value.Value;



/**
 * The parser is used to convert a SQL statement string to an command object.
 */
public class Parser extends Lexer {
	Database database;
	Session session;
	
    protected Prepared prepared;
    protected Prepared currentPrepared;
    protected Select currentSelect;
    protected String schemaName;
    
    public Parser(Database database, Session session){
    	this.database = database;
    	this.session = session;
    }
	
    /**
     * Parse the statement, but don't prepare it for execution.
     * 
     * @param sql the SQL statement to parse
     * @return the prepared object
     */
    public Prepared parseOnly(String sql) throws SQLException {
        try {
            return parse(sql);
        } catch (Exception e) {
            throw Message.convert(e);
        }
    }
    
    private Prepared parse(String sql) throws SQLException {
        initialize(sql);
        expectedList = new ArrayList();
        currentSelect = null;
        currentPrepared = null;
        prepared = null;
        read();
        return parsePrepared();
    }
    
    public Prepared parsePrepared() throws SQLException {
        int start = lastParseIndex;
        Prepared c = null;
        String token = currentToken;
  
		char first = token.charAt(0);
		switch (first) {
		case '(':
		    c = parseSelect();
		    break;
		case 'D':
		    if (readIf("DELETE"))
		        //c = parseDelete();
		    	throw getUnsupportedError();
		case 'F':
		    if (isToken("FROM"))
		        c = parseSelect();
		    break;
		case 'I':
		    if (readIf("INSERT"))
		        //c = parseInsert();
		    	throw getUnsupportedError();
		    break;
		case 'M':
		    if (readIf("MERGE"))
		        //c = parseMerge();
		    	throw getUnsupportedError();
		    break;
		case 'S':
		    if (isToken("SELECT"))
		        c = parseSelect();
		case 'U':
		    if (readIf("UPDATE"))
		        //c = parseUpdate();
		    	throw getUnsupportedError();
		    break;
		default:
		    throw getSyntaxError();
		}

        if (c == null) {
            throw getSyntaxError();
        }
        //setSQL(c, null, start);
        return c;
    }
  
    
    private Query parseSelect() throws SQLException {
        Query command = parseSelectUnion();
        //command.init();
        return command;
    }
    
    private Query parseSelectUnion() throws SQLException {
        int start = lastParseIndex;
        Query command = parseSelectSub();
        return parseSelectUnionExtension(command, start, false);
    }
    
    private Query parseSelectUnionExtension(Query command, int start, boolean unionOnly) throws SQLException {
        while (true) {
            if (readIf("UNION")) {
            	throw getSyntaxError();
            } else if (readIf("MINUS") || readIf("EXCEPT")) {
            	throw getSyntaxError();
            } else if (readIf("INTERSECT")) {
            	throw getSyntaxError();
            } else {
                break;
            }
        }
        if (!unionOnly) {
            parseEndOfQuery(command);
        }
        return command;
    }
    
    private void parseEndOfQuery(Query command) throws SQLException {
        if (readIf("ORDER")) {
        	read("BY");
        	
        	Select oldSelect = currentSelect;
            if (command instanceof Select) {
                currentSelect = (Select) command;
            }
            List orderList = new ArrayList();
            do {
                boolean canBeNumber = true;
                if (readIf("=")) {
                    canBeNumber = false;
                }
                SelectOrderBy order = new SelectOrderBy();
                Expression expr = readExpression();
                if (canBeNumber && expr instanceof ValueExpression && expr.getType() == Value.INT) {
                    order.columnIndexExpr = expr;
                } else if (expr instanceof Parameter) {
                    order.columnIndexExpr = expr;
                } else {
                    order.expression = expr;
                }
                if (readIf("DESC")) {
                    order.descending = true;
                } else {
                    readIf("ASC");
                }
                if (readIf("NULLS")) {
                    if (readIf("FIRST")) {
                        order.nullsFirst = true;
                    } else {
                        read("LAST");
                        order.nullsLast = true;
                    }
                }
                orderList.add(order);
            } while (readIf(","));
            command.setOrder(orderList);
            currentSelect = oldSelect;
        }
        if (supportOffsetFetch) {
            // make sure aggregate functions will not work here
            Select temp = currentSelect;
            currentSelect = null;
            
            // http://sqlpro.developpez.com/SQL2008/
            if (readIf("OFFSET")) {
            	throw getUnsupportedError();
                
            }
            if (readIf("FETCH")) {
                read("FIRST");
                throw getUnsupportedError();
            }
            
            currentSelect = temp;
        }
        if (readIf("LIMIT")) {
            Select temp = currentSelect;
            // make sure aggregate functions will not work here
            currentSelect = null;
            Expression limit = readExpression();
            command.setLimit(limit);
            if (readIf("OFFSET")) {
                Expression offset = readExpression();
                command.setOffset(offset);
            } else if (readIf(",")) {
                // MySQL: [offset, ] rowcount
                Expression offset = limit;
                limit = readExpression();
                command.setOffset(offset);
                command.setLimit(limit);
            }
            if (readIf("SAMPLE_SIZE")) {
            	throw getUnsupportedError(); 
            }
            currentSelect = temp;
            
        }
        if (readIf("FOR")) {
            if (readIf("UPDATE")) {
            	throw getUnsupportedError();
                
            } else if (readIf("READ")) {
            	throw getUnsupportedError(); 
            }
        }
    }

    private Query parseSelectSub() throws SQLException {
        if (readIf("(")) {
            Query command = parseSelectUnion();
            read(")");
            return command;
        }
        Select select = parseSelectSimple();
        return select;
    }

    private void parseSelectSimpleFromPart(Select command) throws SQLException {
        do {
            TableFilter filter = readTableFilter();
            parseJoinTableFilter(filter, command);
        } while (readIf(","));
    }

    private void parseJoinTableFilter(TableFilter top, Select command) throws SQLException {
        top = readJoin(top, command);//, top.isJoinOuter());
        command.setTableFilter(top);
    }

    private void parseSelectSimpleSelectPart(Select command) throws SQLException {
        Select temp = currentSelect;
        // make sure aggregate functions will not work in TOP and LIMIT
        currentSelect = null;
        if (readIf("TOP")) {
            // can't read more complex expressions here because
            // SELECT TOP 1 +? A FROM TEST could mean
            // SELECT TOP (1+?) A FROM TEST or
            // SELECT TOP 1 (+?) AS A FROM TEST
            Expression limit = readTerm();
            command.setLimit(limit);
        } else if (readIf("LIMIT")) {
            Expression offset = readTerm();
            command.setOffset(offset);
            Expression limit = readTerm();
            command.setLimit(limit);
        }
        currentSelect = temp;
        if (readIf("DISTINCT")) {
            throw getUnsupportedError();
        } 
        
        List<Expression> expressions = new ArrayList();
        do {
            if (readIf("*")) {
                expressions.add(new Wildcard());
            } else {
                Expression expr = readExpression();
                if (readIf("AS") || currentTokenType == IDENTIFIER) {
                    String alias = readAliasIdentifier();
                    expr = new Alias(expr, alias);
                }
                expressions.add(expr);
            }
        } while (readIf(","));
        command.setExpressions(expressions);
    }

    private Select parseSelectSimple() throws SQLException {
        boolean fromFirst;
        if (readIf("SELECT")) {
            fromFirst = false;
        } else if (readIf("FROM")) {
            fromFirst = true;
        } else {
            throw getSyntaxError();
        }
        Select command = new Select(session);
        int start = lastParseIndex;
        Select oldSelect = currentSelect;
        currentSelect = command;
        currentPrepared = command;
        if (fromFirst) {
            parseSelectSimpleFromPart(command);
            read("SELECT");
            parseSelectSimpleSelectPart(command);
        } else {
            parseSelectSimpleSelectPart(command);
            if (!readIf("FROM")) {
            	throw getUnsupportedError();
            } else {
                parseSelectSimpleFromPart(command);
            }
        }
        if (readIf("WHERE")) {
            Expression condition = readExpression();
            command.addCondition(condition);
        }
        // the group by is read for the outer select (or not a select)
        // so that columns that are not grouped can be used
        currentSelect = oldSelect;
        if (readIf("GROUP")) {
            read("BY");
            command.setGroupQuery();
            List list = new ArrayList();
            do {
                Expression expr = readExpression();
                list.add(expr);
            } while (readIf(","));
            command.setGroupBy(list);
        }
        currentSelect = command;
        if (readIf("HAVING")) {
            command.setGroupQuery();
            Expression condition = readExpression();
            command.setHaving(condition);
        }
        currentSelect = oldSelect;
        //setSQL(command, "SELECT", start);
        return command;
    }
    
    
    private Expression readExpression() throws SQLException {
        Expression r = readAnd();
        while (readIf("OR")) {
            r = new ConditionAndOr(ConditionAndOr.OR, r, readAnd());
        }
        return r;
    }

    private Expression readAnd() throws SQLException {
        Expression r = readCondition();
        while (readIf("AND")) {
            r = new ConditionAndOr(ConditionAndOr.AND, r, readCondition());
        }
        return r;
    }

    private Expression readCondition() throws SQLException {
        // TODO parser: should probably use switch case for performance
        if (readIf("NOT")) {
            return new ConditionNot(readCondition());
        }
        if (readIf("EXISTS")) {
            read("(");
            Query query = parseSelect();
            // can not reduce expression because it might be a union except
            // query with distinct
            read(")");
            return new ConditionExists(query);
        }
        Expression r = readConcat();
        while (true) {
            // special case: NOT NULL is not part of an expression (as in CREATE
            // TABLE TEST(ID INT DEFAULT 0 NOT NULL))
            int backup = parseIndex;
            boolean not = false;
            if (readIf("NOT")) {
                not = true;
                if (isToken("NULL")) {
                    // this really only works for NOT NULL!
                    parseIndex = backup;
                    currentToken = "NOT";
                    break;
                }
            }
            if (readIf("LIKE")) {
                Expression b = readConcat();
                Expression esc = null;
                if (readIf("ESCAPE")) {
                    esc = readConcat();
                }
                r = new CompareLike(r, b, esc, false);
            } else if (readIf("REGEXP")) {
                Expression b = readConcat();
                r = new CompareLike(r, b, null, true);
            } else if (readIf("IS")) {
                int type;
                if (readIf("NOT")) {
                    type = Comparison.IS_NOT_NULL;
                } else {
                    type = Comparison.IS_NULL;
                }
                read("NULL");
                r = new Comparison(type, r, null);
            } else if (readIf("IN")) {
            	throw getUnsupportedError();
            } else if (readIf("BETWEEN")) {
            	throw getUnsupportedError();
            } else {
                // TODO parser: if we use a switch case, we don't need
                // getCompareType any more
                int compareType = getCompareType(currentTokenType);
                if (compareType < 0) {
                    break;
                }
                read();
                if (readIf("ALL")) {
                	throw getUnsupportedError();
                } else if (readIf("ANY") || readIf("SOME")) {
                	throw getUnsupportedError();
                } else {
                    Expression right = readConcat();
                    r = new Comparison(compareType, r, right);
                }
            }
            if (not) {
                r = new ConditionNot(r);
            }
        }
        return r;
    }
    
    private int getCompareType(int tokenType) {
        switch (tokenType) {
        case EQUAL:
            return Comparison.EQUAL;
        case BIGGER_EQUAL:
            return Comparison.BIGGER_EQUAL;
        case BIGGER:
            return Comparison.BIGGER;
        case SMALLER:
            return Comparison.SMALLER;
        case SMALLER_EQUAL:
            return Comparison.SMALLER_EQUAL;
        case NOT_EQUAL:
            return Comparison.NOT_EQUAL;
        default:
            return -1;
        }
    }
    
    private Expression readConcat() throws SQLException {
        Expression r = readSum();
        while (true) {
            if (readIf("||")) {
            	throw getUnsupportedError();
                //r = new Operation(Operation.CONCAT, r, readSum());
            } else if (readIf("~")) {
            	throw getUnsupportedError();
            } else if (readIf("!~")) {
            	throw getUnsupportedError();
            } else {
                return r;
            }
        }
    }
    

    private Expression readSum() throws SQLException {
        Expression r = readFactor();
        while (true) {
            if (readIf("+")) {
                r = new Operation(Operation.PLUS, r, readFactor());
            } else if (readIf("-")) {
                r = new Operation(Operation.MINUS, r, readFactor());
            } else {
                return r;
            }
        }
    }

    private Expression readFactor() throws SQLException {
        Expression r = readTerm();
        while (true) {
            if (readIf("*")) {
                r = new Operation(Operation.MULTIPLY, r, readTerm());
            } else if (readIf("/")) {
                r = new Operation(Operation.DIVIDE, r, readTerm());
            } else {
                return r;
            }
        }
    }
    
    private Expression readAggregate(int aggregateType) throws SQLException {
        if (currentSelect == null) {
            throw getSyntaxError();
        }
        currentSelect.setGroupQuery();
        Expression r;
        if (aggregateType == Aggregate.COUNT) {
            if (readIf("*")) {
                r = new Aggregate(Aggregate.COUNT_ALL, null, currentSelect, false);
            } else {
                boolean distinct = readIf("DISTINCT");
                Expression on = readExpression();
                if (on instanceof Wildcard && !distinct) {
                    // PostgreSQL compatibility: count(t.*)
                    r = new Aggregate(Aggregate.COUNT_ALL, null, currentSelect, false);
                } else {
                    r = new Aggregate(Aggregate.COUNT, on, currentSelect, distinct);
                }
            }
        } else if (aggregateType == Aggregate.GROUP_CONCAT) {
            boolean distinct = readIf("DISTINCT");
            Aggregate agg = new Aggregate(Aggregate.GROUP_CONCAT, readExpression(), currentSelect, distinct);
            if (readIf("ORDER")) {
                read("BY");
                agg.setOrder(parseSimpleOrderList());
            }
            if (readIf("SEPARATOR")) {
                agg.setSeparator(readExpression());
            }
            r = agg;
        } else {
            boolean distinct = readIf("DISTINCT");
            r = new Aggregate(aggregateType, readExpression(), currentSelect, distinct);
        }
        read(")");
        return r;
    }

    private ArrayList parseSimpleOrderList() throws SQLException {
    	ArrayList orderList = new ArrayList();
        do {
            SelectOrderBy order = new SelectOrderBy();
            Expression expr = readExpression();
            order.expression = expr;
            if (readIf("DESC")) {
                order.descending = true;
            } else {
                readIf("ASC");
            }
            orderList.add(order);
        } while (readIf(","));
        return orderList;
    }

    private Expression readTermObjectDot(String objectName) throws SQLException {
        Expression expr = readWildcardOrSequenceValue(null, objectName);
        if (expr != null) {
            return expr;
        }
        String name = readColumnIdentifier();
        if (readIf(".")) {
            String schema = objectName;
            objectName = name;
            expr = readWildcardOrSequenceValue(schema, objectName);
            if (expr != null) {
                return expr;
            }
            name = readColumnIdentifier();
            if (readIf(".")) {
//                String databaseName = schema;
//                if (!database.getShortName().equals(databaseName)) {
//                    throw Message.getSQLException("DATABASE_NOT_FOUND_1", databaseName, null);
//                }
                schema = objectName;
                objectName = name;
                expr = readWildcardOrSequenceValue(schema, objectName);
                if (expr != null) {
                    return expr;
                }
                name = readColumnIdentifier();
                return new ExpressionColumn(schema, objectName, name);
            }
            return new ExpressionColumn(schema, objectName, name);
        }
        return new ExpressionColumn(null, objectName, name);
    }
    
    private Expression readWildcardOrSequenceValue(String schema, String objectName) throws SQLException {
        if (readIf("*")) {
        	throw getUnsupportedError();
        }
        if (schema == null) {
            //schema = session.getCurrentSchemaName();
        }
        if (readIf("NEXTVAL")) {
        	throw getUnsupportedError();
        } else if (readIf("CURRVAL")) {
        	throw getUnsupportedError();
        }
        return null;
    }
    
    private Expression readTerm() throws SQLException {
        Expression r;
        switch (currentTokenType) {
        case AT:
            read();
            if(true)throw getUnsupportedError();
            break;
        case PARAMETER:
            // there must be no space between ? and the number
            read();
            r = new Parameter();
            break;
        case KEYWORD:
            if (isToken("SELECT") || isToken("FROM")) {
                throw getUnsupportedError();
            } else {
                if(true)throw getSyntaxError();
            }
            break;
        case IDENTIFIER:
            String name = currentToken;
            if (currentTokenQuoted) {
                read();
                if (readIf("(")) {
                    r = readFunction(name);
                } else if (readIf(".")) {
                    r = readTermObjectDot(name);
                } else {
                    r = new ExpressionColumn(null, null, name);
                }
            } else {
                read();
                if (readIf(".")) {
                    r = readTermObjectDot(name);
                } else if ("CASE".equals(name)) {
                    // CASE must be processed before (,
                    // otherwise CASE(3) would be a function call, which it is
                    // not
                	throw getUnsupportedError();

                } else if (readIf("(")) {
                    r = readFunction(name);
                } else {
                    r = new ExpressionColumn(null, null, name);
                }
            }
            break;
        case MINUS:
            read();
            r = new Operation(Operation.NEGATE, readTerm(), null);
            break;
        case PLUS:
            read();
            r = readTerm();
            break;
        case OPEN:
            read();
            r = readExpression();
            if (readIf(",")) {
            	throw getUnsupportedError();
            }
            read(")");
            break;
        case TRUE:
            read();
            r = ValueExpression.TRUE;
            break;
        case FALSE:
            read();
            r = ValueExpression.FALSE;
            break;
        case CURRENT_TIME:
            read();
            if(true) throw getUnsupportedError();
            break;
        case CURRENT_DATE:
            read();
            if(true) throw getUnsupportedError();
            break;
        case CURRENT_TIMESTAMP: {
        	if(true) throw getUnsupportedError();
            break;
        }
        case ROWNUM:
            read();
            if(true) throw getUnsupportedError();
            break;
        case NULL:
            read();
            r = ValueExpression.NULL;
            break;
        case VALUE:
            r = new ValueExpression(currentValue);
            read();
            break;
        default:
            throw getSyntaxError();
        }
        return r;
    }

    private String readAliasIdentifier() throws SQLException {
        return readColumnIdentifier();
    }

    private String readColumnIdentifier() throws SQLException {
        if (currentTokenType != IDENTIFIER) {
            throw Message.getSyntaxError(sqlCommand, parseIndex, "identifier");
        }
        String s = currentToken;
        read();
        return s;
    }
    
    private Expression readFunction(String name) throws SQLException {
        int agg = Aggregate.getAggregateType(name);
        if (agg >= 0) {
            return readAggregate(agg);
        }
        Function function = new Function(name);
        if (!readIf(")")) {
            int i = 0;
            do {
                function.setParameter(i++, readExpression());
            } while (readIf(","));
            read(")");
        }
        return function;
    }
    
    private Schema getSchema() throws SQLException {
        if (schemaName == null) {
            return null;
        }
        Schema schema = database.findSchema(schemaName);
        if (schema == null) {
            if ("SESSION".equals(schemaName)) {
                // for local temporary tables
                schema = database.getSchema(session.getCurrentSchemaName());
            } else {
                throw Message.getSQLException("SCHEMA_NOT_FOUND_1", schemaName);
            }
        }
        return schema;
    }

    
    private String readIdentifierWithSchema(String defaultSchemaName) throws SQLException {
        if (currentTokenType != IDENTIFIER) {
            throw Message.getSyntaxError(sqlCommand, parseIndex, "identifier");
        }
        String s = currentToken;
        read();
        schemaName = defaultSchemaName;
        if (readIf(".")) {
            schemaName = s;
            if (currentTokenType != IDENTIFIER) {
                throw Message.getSyntaxError(sqlCommand, parseIndex, "identifier");
            }
            s = currentToken;
            read();
        }
        if (".".equals(currentToken)) {
            if (schemaName.equalsIgnoreCase(session.getDBShortName())) {
                read(".");
                schemaName = s;
                if (currentTokenType != IDENTIFIER) {
                    throw Message.getSyntaxError(sqlCommand, parseIndex, "identifier");
                }
                s = currentToken;
                read();
            }
        }
        return s;
    }

    private Table readTableOrView(String tableName) throws SQLException {
        // same algorithm than readSequence
        if (schemaName != null) {
            return getSchema().getTableOrView(tableName);
        }
        Table table = database.getSchema(session.getCurrentSchemaName()).findTableOrView(tableName);
        if (table != null) {
            return table;
        }
        return session.findTableOrView(tableName);
    }
    
    private TableFilter readTableFilter(/*boolean fromOuter*/) throws SQLException {
        Table table;
        String alias = null;
        if (readIf("(")) {
            if (isToken("SELECT") || isToken("FROM")) {
                int start = lastParseIndex;
                Query query = parseSelectUnion();
                read(")");
                query = parseSelectUnionExtension(query, start, true);
                //query.init();

//                if (prepared != null && prepared instanceof CreateView) {
//                	throw getUnsupportedError();
//                } else {
//                    s = session;
//                }
                table = TableView.createTempView(session.getNextTempViewName(), query);
                alias = table.getName();
            } else {
                TableFilter top = readTableFilter();//fromOuter);
                top = readJoin(top, currentSelect);//, fromOuter);
                read(")");
                alias = readFromAlias(null);
                if (alias != null) {
                    top.setAlias(alias);
                }
                return top;
            }
        } else {
            String tableName = readIdentifierWithSchema(null);
            if (readIf("(")) {
            	throw getUnsupportedError();
            } else if ("DUAL".equals(tableName)) {
                throw getUnsupportedError();
            } else {
                table = readTableOrView(tableName);
            }
        }
        alias = readFromAlias(alias);
        return new TableFilter(table, alias);
    }
    
    private TableFilter readJoin(TableFilter top, Select command/*, boolean fromOuter*/) throws SQLException {
        while (true) {
            if (readIf("RIGHT")) {
                readIf("OUTER");
                read("JOIN");
                TableFilter join = readTableFilter();
                top = readJoin(top, command);//, true);
                Expression on = null;
                if (readIf("ON")) {
                    on = readExpression();
                }
                top.addJoin(join, "RIGHT", on);
            } else if (readIf("LEFT")) {
                readIf("OUTER");
                read("JOIN");
                TableFilter join = readTableFilter();
                top = readJoin(top, command);//, true);
                Expression on = null;
                if (readIf("ON")) {
                    on = readExpression();
                }
                top.addJoin(join, "LEFT", on);
            } else if (readIf("FULL")) {
                throw this.getSyntaxError();
            } else if (readIf("INNER")) {
                read("JOIN");
                TableFilter join = readTableFilter();
                top = readJoin(top, command);//, true);
                Expression on = null;
                if (readIf("ON")) {
                    on = readExpression();
                }
                top.addJoin(join, "INNER", on);
            } else if (readIf("JOIN")) {
            	 TableFilter join = readTableFilter();
                 top = readJoin(top, command);//, true);
                 Expression on = null;
                 if (readIf("ON")) {
                     on = readExpression();
                 }
                 top.addJoin(join, "", on);
            } else if (readIf("CROSS")) {
            	throw this.getSyntaxError();
            } else if (readIf("NATURAL")) {
            	throw this.getSyntaxError();
            } else {
                break;
            }
        }
        return top;
    }
    private String readFromAlias(String alias) throws SQLException {
        if (readIf("AS")) {
            alias = readAliasIdentifier();
        } else if (currentTokenType == IDENTIFIER) {
            // left and right are not keywords (because they are functions as
            // well)
            if (!isToken("LEFT") && !isToken("RIGHT") && !isToken("FULL")) {
                alias = readAliasIdentifier();
            }
        }
        return alias;
    }
}
