/*
 * Copyright 2004-2008 H2 Group. Multiple-Licensed under the H2 License, 
 * Version 1.0, and under the Eclipse Public License, Version 1.0
 * (http://h2database.com/html/license.html).
 * Initial Developer: H2 Group
 */
package tim.sql.h2parser.expr;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

import tim.sql.h2parser.Message;
import tim.sql.h2parser.ObjectUtils;
import tim.sql.h2parser.StringUtils;
import tim.sql.h2parser.dml.Select;
import tim.sql.h2parser.dml.SelectOrderBy;


/**
 * Implements the integrated aggregate functions, such as COUNT, MAX, SUM.
 */
public class Aggregate extends Expression {
    
    /**
     * The aggregate type for COUNT(*).
     */
    public static final int COUNT_ALL = 0;
    
    /**
     * The aggregate type for COUNT(expression).
     */
    public static final int COUNT = 1;
    
    /**
     * The aggregate type for SUM(expression).
     */
    public static final int SUM = 2;
    
    /**
     * The aggregate type for MIN(expression).
     */
    public static final int MIN = 3;

    /**
     * The aggregate type for MAX(expression).
     */
    public static final int MAX = 4;
    
    /**
     * The aggregate type for AVG(expression).
     */
    public static final int AVG = 5;
    
    /**
     * The aggregate type for GROUP_CONCAT(...).
     */
    public static final int GROUP_CONCAT = 6;
    
    /**
     * The aggregate type for STDDEV_POP(expression).
     */
    public static final int STDDEV_POP = 7;
    
    /**
     * The aggregate type for STDDEV_SAMP(expression).
     */
    public static final int STDDEV_SAMP = 8;
    
    /**
     * The aggregate type for VAR_POP(expression).
     */
    public static final int VAR_POP = 9;
    
    /**
     * The aggregate type for VAR_SAMP(expression).
     */
    public static final int VAR_SAMP = 10;
    
    /**
     * The aggregate type for BOOL_OR(expression).
     */
    public static final int BOOL_OR = 11;
    
    /**
     * The aggregate type for BOOL_AND(expression).
     */
    public static final int BOOL_AND = 12;
    
    /**
     * The aggregate type for SELECTIVITY(expression).
     */
    public static final int SELECTIVITY = 13;
    
    private static final HashMap AGGREGATES = new HashMap();

    private final int type;
    private final Select select;
    private final boolean distinct;

    private Expression on;
    private Expression separator;
    private List<SelectOrderBy> orderList;
    //private SortOrder sort;
    private int dataType, scale;
    private long precision;
    private int displaySize;
    private int lastGroupRowId;

    /**
     * Create a new aggregate object.
     * 
     * @param type the aggregate type
     * @param on the aggregated expression
     * @param select the select statement
     * @param distinct if distinct is used
     */
    public Aggregate(int type, Expression on, Select select, boolean distinct) {
        this.type = type;
        this.on = on;
        this.select = select;
        this.distinct = distinct;
    }

    static {
        addAggregate("COUNT", COUNT);
        addAggregate("SUM", SUM);
        addAggregate("MIN", MIN);
        addAggregate("MAX", MAX);
        addAggregate("AVG", AVG);
        addAggregate("GROUP_CONCAT", GROUP_CONCAT);
        addAggregate("STDDEV_SAMP", STDDEV_SAMP);
        addAggregate("STDDEV", STDDEV_SAMP);
        addAggregate("STDDEV_POP", STDDEV_POP);
        addAggregate("STDDEVP", STDDEV_POP);
        addAggregate("VAR_POP", VAR_POP);
        addAggregate("VARP", VAR_POP);
        addAggregate("VAR_SAMP", VAR_SAMP);
        addAggregate("VAR", VAR_SAMP);
        addAggregate("VARIANCE", VAR_SAMP);
        addAggregate("BOOL_OR", BOOL_OR);
        // HSQLDB compatibility, but conflicts with >EVERY(...)
        addAggregate("SOME", BOOL_OR);
        addAggregate("BOOL_AND", BOOL_AND);
        // HSQLDB compatibility, but conflicts with >SOME(...)
        addAggregate("EVERY", BOOL_AND);
        addAggregate("SELECTIVITY", SELECTIVITY);
    }

    private static void addAggregate(String name, int type) {
        AGGREGATES.put(name, ObjectUtils.getInteger(type));
    }

    /**
     * Get the aggregate type for this name, or -1 if no aggregate has been
     * found.
     * 
     * @param name the aggregate function name
     * @return -1 if no aggregate function has been found, or the aggregate type
     */
    public static int getAggregateType(String name) {
        Integer type = (Integer) AGGREGATES.get(name);
        return type == null ? -1 : type.intValue();
    }

    /**
     * Set the order for GROUP_CONCAT.
     * 
     * @param orderBy the order by list
     */
    public void setOrder(ArrayList orderBy) {
        this.orderList = orderBy;
    }

    /**
     * Set the separator for GROUP_CONCAT.
     * 
     * @param separator the separator expression
     */
    public void setSeparator(Expression separator) {
        this.separator = separator;
    }


    public int getType() {
        return dataType;
    }


    public int getScale() {
        return scale;
    }

    public long getPrecision() {
        return precision;
    }

    public int getDisplaySize() {
        return displaySize;
    }

    
    
    private String getSQLGroupConcat() {
        StringBuffer buff = new StringBuffer();
        buff.append("GROUP_CONCAT(");
        buff.append(on.getSQL());
        if (orderList != null) {
            buff.append(" ORDER BY ");
            for (int i = 0; i < orderList.size(); i++) {
                SelectOrderBy o = (SelectOrderBy) orderList.get(i);
                if (i > 0) {
                    buff.append(", ");
                }
                buff.append(o.expression.getSQL());
                if (o.descending) {
                    buff.append(" DESC");
                }
            }
        }
        if (separator != null) {
            buff.append(" SEPARATOR ");
            buff.append(separator.getSQL());
        }
        buff.append(")");
        return buff.toString();
    }
    
    public String getSQL() {
        String text;
        switch (type) {
        case GROUP_CONCAT:
            return getSQLGroupConcat();
        case COUNT_ALL:
            return "COUNT(*)";
        case COUNT:
            text = "COUNT";
            break;
        case SELECTIVITY:
            text = "SELECTIVITY";
            break;
        case SUM:
            text = "SUM";
            break;
        case MIN:
            text = "MIN";
            break;
        case MAX:
            text = "MAX";
            break;
        case AVG:
            text = "AVG";
            break;
        case STDDEV_POP:
            text = "STDDEV_POP";
            break;
        case STDDEV_SAMP:
            text = "STDDEV_SAMP";
            break;
        case VAR_POP:
            text = "VAR_POP";
            break;
        case VAR_SAMP:
            text = "VAR_SAMP";
            break;
        case BOOL_AND:
            text = "BOOL_AND";
            break;
        case BOOL_OR:
            text = "BOOL_OR";
            break;
        default:
            throw Message.getInternalError("type=" + type);
        }
        if (distinct) {
            return text + "(DISTINCT " + on.getSQL() + ")";
        }
        return text + StringUtils.enclose(on.getSQL());
    }
    
	@Override
	public void buildColList(List<ExpressionColumn> r) {
		on.buildColList(r);
		if(separator!=null) separator.buildColList(r);
		if(orderList!=null){ 
			for (int i = 0; i < orderList.size(); i++) {
				SelectOrderBy o = (SelectOrderBy) orderList.get(i);
				o.expression.buildColList(r);
			}
		}
	}

    
}
