/*
 * Copyright 2004-2008 H2 Group. Multiple-Licensed under the H2 License, 
 * Version 1.0, and under the Eclipse Public License, Version 1.0
 * (http://h2database.com/html/license.html).
 * Initial Developer: H2 Group
 */
package tim.sql.h2parser.expr;

import java.util.List;

import tim.sql.h2parser.Message;


/**
 * Example comparison expressions are ID=1, NAME=NAME, NAME IS NULL.
 */
public class Comparison extends Condition {

    /**
     * The comparison type meaning = as in ID=1.
     */
    public static final int EQUAL = 0;
    
    /**
     * The comparison type meaning &gt;= as in ID&gt;=1.
     */
    public static final int BIGGER_EQUAL = 1;
    
    /**
     * The comparison type meaning &gt; as in ID&gt;1.
     */
    public static final int BIGGER = 2;

    /**
     * The comparison type meaning &lt;= as in ID&lt;=1.
     */
    public static final int SMALLER_EQUAL = 3;
    
    /**
     * The comparison type meaning &lt; as in ID&lt;1.
     */
    public static final int SMALLER = 4;
    
    /**
     * The comparison type meaning &lt;&gt; as in ID&lt;&gt;1.
     */
    public static final int NOT_EQUAL = 5;
    
    /**
     * The comparison type meaning IS NULL as in NAME IS NULL.
     */
    public static final int IS_NULL = 6;
    
    /**
     * The comparison type meaning IS NOT NULL as in NAME IS NOT NULL.
     */
    public static final int IS_NOT_NULL = 7;

    /**
     * This is a pseudo comparison type that is only used for index conditions.
     * It means the comparison will always yield FALSE. Example: 1=0.
     */
    public static final int FALSE = 8;

    private final int compareType;
    private Expression left;
    private Expression right;
    private int dataType = -2;

    public Comparison(int compareType, Expression left, Expression right) {
        this.left = left;
        this.right = right;
        this.compareType = compareType;
    }

    public String getSQL() {
        String sql;
        switch (compareType) {
        case EQUAL:
            sql = left.getSQL() + " = " + right.getSQL();
            break;
        case BIGGER_EQUAL:
            sql = left.getSQL() + " >= " + right.getSQL();
            break;
        case BIGGER:
            sql = left.getSQL() + " > " + right.getSQL();
            break;
        case SMALLER_EQUAL:
            sql = left.getSQL() + " <= " + right.getSQL();
            break;
        case SMALLER:
            sql = left.getSQL() + " < " + right.getSQL();
            break;
        case NOT_EQUAL:
            sql = left.getSQL() + " <> " + right.getSQL();
            break;
        case IS_NULL:
            sql = left.getSQL() + " IS NULL";
            break;
        case IS_NOT_NULL:
            sql = left.getSQL() + " IS NOT NULL";
            break;
        default:
            throw Message.getInternalError("compareType=" + compareType);
        }
        return "(" + sql + ")";
    }

    /**
     * Get the left or the right sub-expression of this condition.
     * 
     * @param left true to get the left sub-expression, false to get the right
     *            sub-expression.
     * @return the sub-expression
     */
    public Expression getExpression(boolean left) {
        return left ? this.left : right;
    }
    
	public void buildColList(List<ExpressionColumn> r){
		left.buildColList(r);
		// If IS NULL -> right==null
		if(right!=null) right.buildColList(r);
	}

}
