/*
 * Copyright 2004-2008 H2 Group. Multiple-Licensed under the H2 License, 
 * Version 1.0, and under the Eclipse Public License, Version 1.0
 * (http://h2database.com/html/license.html).
 * Initial Developer: H2 Group
 */
package tim.sql.h2parser.expr;

import java.util.List;

import tim.sql.h2parser.Message;


/**
 * An 'and' or 'or' condition as in WHERE ID=1 AND NAME=?
 */
public class ConditionAndOr extends Condition {

    /**
     * The AND condition type as in ID=1 AND NAME='Hello'.
     */
    public static final int AND = 0;

    /**
     * The OR condition type as in ID=1 OR NAME='Hello'.
     */
    public static final int OR = 1;

    private final int andOrType;
    private Expression left, right;

    public ConditionAndOr(int andOrType, Expression left, Expression right) {
        this.andOrType = andOrType;
        this.left = left;
        this.right = right;
        assert(left != null && right != null);
    }

    public String getSQL() {
        String sql;
        switch (andOrType) {
        case AND:
            sql = left.getSQL() + " AND " + right.getSQL();
            break;
        case OR:
            sql = left.getSQL() + " OR " + right.getSQL();
            break;
        default:
            throw Message.getInternalError("andOrType=" + andOrType);
        }
        return "(" + sql + ")";
    }

  

    /**
     * Get the left or the right sub-expression of this condition.
     * 
     * @param left true to get the left sub-expression, false to get the right
     *            sub-expression.
     * @return the sub-expression
     */
    public Expression getExpression(boolean left) {
        return left ? this.left : right;
    }

	public void buildColList(List<ExpressionColumn> r){
		left.buildColList(r);
		right.buildColList(r);
	}
}
