/*
 * Copyright 2004-2008 H2 Group. Multiple-Licensed under the H2 License, 
 * Version 1.0, and under the Eclipse Public License, Version 1.0
 * (http://h2database.com/html/license.html).
 * Initial Developer: H2 Group
 */
package tim.sql.h2parser.value;

import java.sql.PreparedStatement;
import java.sql.SQLException;

/**
 * Implementation of the INT data type.
 */
public class ValueInt extends Value {
    
    /**
     * The precision in digits.
     */
    public static final int PRECISION = 10;
    
    /**
     * The maximum display size of an int.
     * Example: -2147483648
     */
    public static final int DISPLAY_SIZE = 11; 

    private static final int STATIC_SIZE = 100;
    // must be a power of 2
    private static final int DYNAMIC_SIZE = 256;
    // TODO check performance of final static
    private static ValueInt[] staticCache;
    private static ValueInt[] dynamicCache;

    private final int value;
    
    static {
        staticCache = new ValueInt[STATIC_SIZE];
        dynamicCache = new ValueInt[DYNAMIC_SIZE];
        for (int i = 0; i < STATIC_SIZE; i++) {
            staticCache[i] = new ValueInt(i);
        }
    }
    
    private ValueInt(int value) {
        this.value = value;
    }

    /**
     * Get or create an int value for the given int.
     * 
     * @param i the int
     * @return the value
     */
    public static ValueInt get(int i) {
        if (i >= 0 && i < STATIC_SIZE) {
            return staticCache[i];
        }
        ValueInt v = dynamicCache[i & DYNAMIC_SIZE - 1];
        if (v == null || v.value != i) {
            v = new ValueInt(i);
            dynamicCache[i & DYNAMIC_SIZE - 1] = v;
        }
        return v;
    }

    public int getSignum() {
        return value == 0 ? 0 : (value < 0 ? -1 : 1);
    }

    public String getSQL() {
        return getString();
    }

    public int getType() {
        return Value.INT;
    }

    public int getInt() {
        return value;
    }

    public String getString() {
        return String.valueOf(value);
    }

    public long getPrecision() {
        return PRECISION;
    }

    public int hashCode() {
        return value;
    }
    
    public void set(PreparedStatement prep, int parameterIndex) throws SQLException {
        prep.setInt(parameterIndex, value);
    }

    public int getDisplaySize() {
        return DISPLAY_SIZE;
    }

    public boolean equals(Object other) {
        return other instanceof ValueInt && value == ((ValueInt) other).value;
    }

}
