/*
 * Decompiled with CFR 0.152.
 */
package org.h2.test.store;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.Random;
import java.util.TreeMap;
import java.util.concurrent.atomic.AtomicInteger;
import org.h2.mvstore.FileStore;
import org.h2.mvstore.MVMap;
import org.h2.mvstore.MVStore;
import org.h2.mvstore.StreamStore;
import org.h2.store.fs.FileUtils;
import org.h2.test.TestBase;
import org.h2.util.IOUtils;
import org.h2.util.StringUtils;

public class TestStreamStore
extends TestBase {
    public static void main(String ... a) throws Exception {
        TestBase.createCaller().init().testFromMain();
    }

    @Override
    public void test() throws IOException {
        FileUtils.createDirectories(this.getBaseDir());
        this.testMaxBlockKey();
        this.testIOException();
        this.testSaveCount();
        this.testExceptionDuringStore();
        this.testReadCount();
        this.testLarge();
        this.testDetectIllegalId();
        this.testTreeStructure();
        this.testFormat();
        this.testWithExistingData();
        this.testWithFullMap();
        this.testLoop();
    }

    private void testMaxBlockKey() throws IOException {
        TreeMap<Long, byte[]> map = new TreeMap<Long, byte[]>();
        StreamStore s = new StreamStore(map, 64, 128);
        map.clear();
        int len = 1;
        while (len < 0x100000) {
            byte[] id = s.put(new ByteArrayInputStream(new byte[len]));
            long max = s.getMaxBlockKey(id);
            if (max == -1L) {
                this.assertTrue(map.isEmpty());
            } else {
                this.assertEquals(map.lastKey(), (Object)max);
            }
            len *= 2;
        }
    }

    private void testIOException() throws IOException {
        HashMap<Long, byte[]> map = new HashMap<Long, byte[]>();
        StreamStore s = new StreamStore(map);
        byte[] id = s.put(new ByteArrayInputStream(new byte[0x100000]));
        InputStream in = s.get(id);
        map.clear();
        try {
            while (in.read() >= 0) {
            }
            this.fail();
        }
        catch (IOException e) {
            TestStreamStore.checkErrorCode(50, e.getCause());
        }
    }

    private void testSaveCount() throws IOException {
        String fileName = this.getBaseDir() + "/testSaveCount.h3";
        FileUtils.delete(fileName);
        MVStore s = new MVStore.Builder().fileName(fileName).open();
        MVMap<Long, byte[]> map = s.openMap("data");
        StreamStore streamStore = new StreamStore(map);
        int blockSize = 262144;
        this.assertEquals((long)blockSize, streamStore.getMaxBlockSize());
        int i = 0;
        while (i < 128) {
            streamStore.put(new RandomStream(blockSize, i));
            ++i;
        }
        s.close();
        long writeCount = s.getFileStore().getWriteCount();
        this.assertTrue(writeCount > 5L);
    }

    private void testExceptionDuringStore() {
        HashMap<Long, byte[]> map = new HashMap<Long, byte[]>();
        StreamStore s = new StreamStore(map, 256, 1024);
        this.assertThrows(IOException.class, () -> s.put(TestStreamStore.createFailingStream(new IOException())));
        this.assertEquals(0, map.size());
        this.assertThrows(IOException.class, () -> s.put(TestStreamStore.createFailingStream(new IllegalStateException())));
        this.assertEquals(0, map.size());
    }

    private void testReadCount() throws IOException {
        String fileName = this.getBaseDir() + "/testReadCount.h3";
        FileUtils.delete(fileName);
        MVStore s = new MVStore.Builder().fileName(fileName).open();
        FileStore<?> fileStore = s.getFileStore();
        fileStore.setCacheSize(1);
        StreamStore streamStore = TestStreamStore.getAutoCommitStreamStore(s);
        long size = fileStore.getMaxPageSize() * 2L;
        int i = 0;
        while (i < 100) {
            streamStore.put(new RandomStream(size, i));
            ++i;
        }
        s.commit();
        MVMap map = s.openMap("data");
        this.assertTrue("size: " + map.size(), map.sizeAsLong() >= 100L);
        s.close();
        s = new MVStore.Builder().fileName(fileName).open();
        streamStore = TestStreamStore.getAutoCommitStreamStore(s);
        int i2 = 0;
        while (i2 < 100) {
            streamStore.put(new RandomStream(size, -i2));
            ++i2;
        }
        s.commit();
        long readCount = fileStore.getReadCount();
        this.assertTrue("rc: " + readCount, readCount <= 20L);
        map = s.openMap("data");
        this.assertTrue("size: " + map.size(), map.sizeAsLong() >= 200L);
        s.close();
    }

    private static StreamStore getAutoCommitStreamStore(MVStore s) {
        MVMap<Long, byte[]> map = s.openMap("data");
        return new StreamStore(map, len -> {
            if (s.getUnsavedMemory() > s.getAutoCommitMemory() / 2) {
                s.commit();
            }
        });
    }

    private void testLarge() throws IOException {
        String fileName = this.getBaseDir() + "/testVeryLarge.h3";
        FileUtils.delete(fileName);
        MVStore s = new MVStore.Builder().fileName(fileName).open();
        MVMap<Long, byte[]> map = s.openMap("data");
        AtomicInteger count = new AtomicInteger();
        StreamStore streamStore = new StreamStore(map, len -> {
            count.incrementAndGet();
            s.commit();
        });
        streamStore.put(new RandomStream(0x100000L, 0));
        s.close();
        this.assertEquals(4, count.get());
    }

    private void testDetectIllegalId() {
        HashMap<Long, byte[]> map = new HashMap<Long, byte[]>();
        StreamStore store = new StreamStore(map);
        this.assertThrows(IllegalArgumentException.class, () -> {
            byte[] byArray = new byte[3];
            byArray[0] = 3;
            return store.length(byArray);
        });
        this.assertThrows(IllegalArgumentException.class, () -> {
            byte[] byArray = new byte[3];
            byArray[0] = 3;
            store.remove(byArray);
        });
        byte[] byArray = new byte[3];
        byArray[0] = 3;
        map.put(0L, byArray);
        byte[] byArray2 = new byte[3];
        byArray2[0] = 2;
        byArray2[1] = 1;
        InputStream in = store.get(byArray2);
        this.assertThrows(IllegalArgumentException.class, () -> in.read());
    }

    private void testTreeStructure() throws IOException {
        final AtomicInteger reads = new AtomicInteger();
        HashMap<Long, byte[]> map = new HashMap<Long, byte[]>(){
            private static final long serialVersionUID = 1L;

            @Override
            public byte[] get(Object k) {
                reads.incrementAndGet();
                return (byte[])super.get(k);
            }
        };
        StreamStore store = new StreamStore((Map<Long, byte[]>)map, 10, 100);
        byte[] id = store.put(new ByteArrayInputStream(new byte[10000]));
        InputStream in = store.get(id);
        this.assertEquals(0, in.read(new byte[0]));
        this.assertEquals(0, in.read());
        this.assertEquals(3, reads.get());
    }

    private void testFormat() throws IOException {
        HashMap<Long, byte[]> map = new HashMap<Long, byte[]>();
        StreamStore store = new StreamStore(map, 10, 20);
        store.setNextKey(123L);
        byte[] id = store.put(new ByteArrayInputStream(new byte[200]));
        this.assertEquals(200L, store.length(id));
        this.assertEquals("02c8018801", StringUtils.convertBytesToHex(id));
        id = store.put(new ByteArrayInputStream(new byte[0]));
        this.assertEquals("", StringUtils.convertBytesToHex(id));
        id = store.put(new ByteArrayInputStream(new byte[1]));
        this.assertEquals("000100", StringUtils.convertBytesToHex(id));
        id = store.put(new ByteArrayInputStream(new byte[3]));
        this.assertEquals("0003000000", StringUtils.convertBytesToHex(id));
        id = store.put(new ByteArrayInputStream(new byte[10]));
        this.assertEquals("010a8901", StringUtils.convertBytesToHex(id));
        byte[] combined = StringUtils.convertHexToBytes("0001aa0002bbcc");
        this.assertEquals(3L, store.length(combined));
        InputStream in = store.get(combined);
        this.assertEquals(1L, in.skip(1L));
        this.assertEquals(187, in.read());
        this.assertEquals(1L, in.skip(1L));
    }

    private void testWithExistingData() throws IOException {
        final AtomicInteger tests = new AtomicInteger();
        HashMap<Long, byte[]> map = new HashMap<Long, byte[]>(){
            private static final long serialVersionUID = 1L;

            @Override
            public boolean containsKey(Object k) {
                tests.incrementAndGet();
                return super.containsKey(k);
            }
        };
        StreamStore store = new StreamStore((Map<Long, byte[]>)map, 10, 20);
        int i = 0;
        while (i < 10) {
            store.put(new ByteArrayInputStream(new byte[20]));
            ++i;
        }
        this.assertEquals(10, map.size());
        this.assertEquals(10, tests.get());
        i = 0;
        while (i < 10) {
            this.assertTrue(map.containsKey(i));
            ++i;
        }
        this.assertEquals(20, tests.get());
        store = new StreamStore((Map<Long, byte[]>)map, 10, 20);
        this.assertEquals(0L, store.getNextKey());
        i = 0;
        while (i < 5) {
            store.put(new ByteArrayInputStream(new byte[20]));
            ++i;
        }
        this.assertEquals(88, tests.get());
        this.assertEquals(15L, store.getNextKey());
        this.assertEquals(15, map.size());
        i = 0;
        while (i < 15) {
            this.assertTrue(map.containsKey(i));
            ++i;
        }
    }

    private void testWithFullMap() throws IOException {
        final AtomicInteger tests = new AtomicInteger();
        HashMap<Long, byte[]> map = new HashMap<Long, byte[]>(){
            private static final long serialVersionUID = 1L;

            @Override
            public boolean containsKey(Object k) {
                tests.incrementAndGet();
                if ((Long)k < 0x3FFFFFFFFFFFFFFFL) {
                    return true;
                }
                return super.containsKey(k);
            }
        };
        StreamStore store = new StreamStore((Map<Long, byte[]>)map, 20, 100);
        store.put(new ByteArrayInputStream(new byte[100]));
        this.assertEquals(1, map.size());
        this.assertEquals(64, tests.get());
        this.assertEquals(0x4000000000000000L, store.getNextKey());
    }

    private void testLoop() throws IOException {
        this.test(10, 20, 1000);
        int i = 0;
        while (i < 20) {
            this.test(0, 128, i);
            this.test(10, 128, i);
            ++i;
        }
        i = 20;
        while (i < 200) {
            this.test(0, 128, i);
            this.test(10, 128, i);
            i += 10;
        }
    }

    private void test(int minBlockSize, int maxBlockSize, int length) throws IOException {
        HashMap<Long, byte[]> map = new HashMap<Long, byte[]>();
        StreamStore store = new StreamStore(map, minBlockSize, maxBlockSize);
        this.assertEquals(minBlockSize, store.getMinBlockSize());
        this.assertEquals((long)maxBlockSize, store.getMaxBlockSize());
        long next = store.getNextKey();
        Random r = new Random(length);
        byte[] data = new byte[length];
        r.nextBytes(data);
        byte[] id = store.put(new ByteArrayInputStream(data));
        if (length > 0 && length >= minBlockSize) {
            this.assertFalse(store.isInPlace(id));
        } else {
            this.assertTrue(store.isInPlace(id));
        }
        long next2 = store.getNextKey();
        if (length > 0 && length >= minBlockSize) {
            this.assertTrue(next2 > next);
        } else {
            this.assertEquals(next, next2);
        }
        if (length == 0) {
            this.assertEquals(0, id.length);
        }
        this.assertEquals((long)length, store.length(id));
        InputStream in = store.get(id);
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        IOUtils.copy(in, (OutputStream)out);
        this.assertTrue(Arrays.equals(data, out.toByteArray()));
        in = store.get(id);
        in.close();
        this.assertEquals(-1, in.read());
        in = store.get(id);
        this.assertEquals(0L, in.skip(0L));
        if (length > 0) {
            this.assertEquals(1L, in.skip(1L));
            if (length > 1) {
                this.assertEquals(data[1] & 0xFF, in.read());
                if (length > 2) {
                    this.assertEquals(1L, in.skip(1L));
                    if (length > 3) {
                        this.assertEquals(data[3] & 0xFF, in.read());
                    }
                } else {
                    this.assertEquals(0L, in.skip(1L));
                }
            } else {
                this.assertEquals(-1, in.read());
            }
        } else {
            this.assertEquals(0L, in.skip(1L));
        }
        if (length > 12) {
            long s;
            in = store.get(id);
            this.assertEquals(12L, in.skip(12L));
            this.assertEquals(data[12] & 0xFF, in.read());
            long skipped = 0L;
            while ((s = in.skip(Integer.MAX_VALUE)) != 0L) {
                skipped += s;
            }
            this.assertEquals((long)(length - 13), skipped);
            this.assertEquals(-1, in.read());
        }
        store.remove(id);
        this.assertEquals(0, store.getMap().size());
    }

    static class RandomStream
    extends InputStream {
        private final long size;
        private long pos;
        private int seed;

        RandomStream(long size, int seed) {
            this.size = size;
            this.seed = seed;
        }

        @Override
        public int read() {
            byte[] data = new byte[1];
            int len = this.read(data, 0, 1);
            return len <= 0 ? len : data[0] & 0xFF;
        }

        @Override
        public int read(byte[] b, int off, int len) {
            if (this.pos >= this.size) {
                return -1;
            }
            len = (int)Math.min(this.size - this.pos, (long)len);
            int x = this.seed;
            int end = off + len;
            while (off < end) {
                x = (x << 4) + x + 1;
                b[off++] = (byte)(x >> 8);
            }
            this.seed = x;
            this.pos += (long)len;
            return len;
        }
    }
}

