package ebuild.api.log;

/**
 * The logging interface. 
 * 
 * <p>
 * Loggers exist in a hierarchy which exists for two reasons, presentation and control of the output.
 * At any point in time an individual logger is in 'charge' and will capture all output System.out and
 * System.err streams. This is for representing (titling and indenting) the output from 3rd party tools
 * when they are invoked.
 * </p>
 * 
 * <p>
 * Indenting is increased by adding named 'sub' loggers, which are generally for displaying contained 
 * sections in a process. For example when calling a compiler there is potentially  a lot of output that 
 * must be forwarded to the console, but as we do not want to just forward it inline we give it a sub 
 * logger. See method prime(). 
 * </p>
 * 
 * <p>
 * Output to a logger can be controlled by providing a switch which it will check before logging anything.
 * </p>
 * 
 * <p>
 * A logging hierarchy is intended for single threaded use only. Multiple threads may access a single
 * logger in the hierarchy, but asynchronous access to different loggers will cause presentation issues.
 * </p>
 * 
 * 
 * @author mikegoodwin
 *
 */
public interface ILogger {
    
    /** The logging system captures the standard out and standard err that occurs during invocations of 3rd
     * libraries. Calling prime() before such an invocation will cause the output to go to this logger (and 
     * not, as it otherwise would to the previously active logger). It is not necessary to call this if
     * the logger was the last logger to be used.
     * */
    public void prime();
    
    /** Log a line to the logger (newline will be added to the end) */
    public void log(String msg);
    public void log(Throwable t);
    
    /** Log raw characters to the logger, (and if console will flush, to display immediately) */
    public void logPartial(String chars);
    
    /** For displaying a header only when there is output. The message will precede any subsequent logs to the logger. 
     * Set to null to remove a potentially active provisional header. */
    public void setProvisionalHeading(String msg);
    
    /** If this loggers output is being used */
    public boolean isOn();
    
    public ILogger newSubLogger(String name);
    public ILogger newSubLogger(ISwitch onSwitch);
    public ILogger newSubLogger(String name, ISwitch onSwitch);
}
