package ebuild.build.java.compile;

import static ebuild.util.StringUtil.join;

import java.io.File;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import org.apache.tools.ant.Project;
import org.apache.tools.ant.types.FileSet;
import org.apache.tools.ant.types.Path;
import org.apache.tools.ant.types.selectors.FilenameSelector;

import ebuild.ant.AntStuff;
import ebuild.ant.AntStuff.Execution;
import ebuild.api.IBuilderArgument;
import ebuild.api.IEBuild;
import ebuild.api.IModuleBuildProject;
import ebuild.api.IModuleRelease;
import ebuild.api.common.Scope;
import ebuild.api.log.ILogger;
import ebuild.util.StringUtil;

public class JavacRunner {
	final IBuilderArgument argument;
	final IEBuild ebuild;
	final ILogger logger;
	final IModuleBuildProject module;
	final Scope scope;
	final boolean debugInformation;
	//final BuildMode mode;
	
	public JavacRunner(IBuilderArgument argument) {
		this.argument = argument;
		this.ebuild = argument.getEBuild();
		this.logger = argument.getLogger();
		this.module = argument.getModuleBuildProject();
		this.scope = argument.getScope();
		this.debugInformation = argument.getPropertyMap().getBool("debug", true);
	}

	protected String getSourceType(){ return "java"; }
	protected String getFilenamePattern(){ return "**/*.java"; }
	
	private List<String> compilationSources(){
		List<File> fs = argument.getInputDirs(getSourceType());
	    List<String> r = new ArrayList(fs.size());
	    for(File f: fs) r.add(f.getAbsolutePath());
	    return r;
	}

	public void buildClassPath(Collection<? extends IModuleRelease> deps, Scope scope, List<String> elements){
        for(IModuleRelease d: deps){
            buildClassPath(d, scope, elements);
        }
    }
	
    public void buildClassPath(IModuleRelease mod, Scope scope, List<String> elements){
        for(Scope s= scope; s!=null; s=s.getParent()){
        	for(File f: mod.getArtifacts(s,"java.classes.jar")){
                elements.add(f.getAbsolutePath());
            }
        }
        buildClassPath(mod.getDependencies(scope), scope, elements);
    }

    
    public List<String> getClassPath(Scope scope) {
        List<String> elements = new ArrayList();
        
        IModuleRelease release = module.getModuleRelease();
        // include classes from parent scopes
        for(Scope s= scope.getParent(); s!=null; s=s.getParent()){
        	for(File f: release.getArtifacts(s,"java.classes.jar")){
                elements.add(f.getAbsolutePath());
            }
        }
        
        for(File l: module.getCheckedInArtifacts(scope,"java.classes.jar")){
            elements.add(l.getAbsolutePath());
        }
        
        
        buildClassPath(module.getReleasedDependencies(scope), scope, elements);
        return elements;
    }
    
    private String compilationClasspath(){
        List<String> classpathList = getClassPath(Scope.MAIN);
        {
            logger.log("classpath: ");
            for(String s: classpathList){
                logger.log("    "+ebuild.formatAsDisplayPath(new File(s)));
            }
        }
        return StringUtil.join(";", classpathList);
    }
    
    private Path sourcePath(Project p, List<String> sources, boolean negate){
        Path sourcePath = new Path(p);
        FilenameSelector fns = new FilenameSelector();
        fns.setName(getFilenamePattern());
        fns.setNegate(negate);
        for(String s: sources){
            FileSet fs = new FileSet();
            fs.setDir(new File(s));
            fs.addFilename(fns);
            sourcePath.addFileset(fs);
        }
        return sourcePath;
    }
    
	
    private void setSpecificProperties(Project p){
        // FEATURE -  make this part configurable, depending on the configuration of the builder
    	String compiler = "ecj";
    	if("ecj".equals(compiler)){
    	   	p.setUserProperty("nowarn", "-nowarn:-unchecked,-serial");
        	p.setUserProperty("build.compiler","org.eclipse.jdt.core.JDTCompilerAdapter");
    	}else if("javac".equals(compiler)){
    		p.setUserProperty("nowarn", "-Xlint:-nocheck,-serial");
    	}
    }
    
	public void doBuild() throws Exception{
	    // options
	   
	    
	    File targetClasses = argument.getOutputDir("java.classes");
	    
        // inputs
	    List<String> sources = compilationSources();
	    logger.log("sources: ");
        for(String s: sources){
            logger.log("    "+ebuild.formatAsDisplayPath(new File(s)));
        }
    
	    if(sources.size()>0){
	    
	        String classpath = compilationClasspath();
        
    	    Project p = new Project();
    	    
    	    setSpecificProperties(p); 
    	    
            p.setUserProperty("output.debug", debugInformation+"");
            p.setUserProperty("includeantruntime", "false");
    
    	    // inputs
            Path resourcePath = sourcePath(p, sources, true);
            Path sourcePath = sourcePath(p, sources, false);

                    
            p.addReference("resource.path", resourcePath);
            p.addReference("source.path", sourcePath);
            p.setUserProperty("src.list", join(";", sources));
            p.setUserProperty("classpath", classpath);
            
            // outputs
            
            targetClasses.mkdirs();
            p.setUserProperty("target.classes", targetClasses.getAbsolutePath());
    		
    		Execution e = AntStuff.newExecution(p, logger, "build.xml", getClass());
            e.executeTarget("copy");
            e.executeTarget("compile");
	    }
    }
}
