package ebuild.build.java.compile;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.antlr.stringtemplate.StringTemplate;
import org.antlr.stringtemplate.StringTemplateGroup;
import org.antlr.stringtemplate.language.DefaultTemplateLexer;

import ebuild.api.IEBuild;
import ebuild.api.IGenConfArgument;
import ebuild.api.IModuleProject;
import ebuild.api.IModuleRelease;
import ebuild.api.IModuleWorkspaceProject;
import ebuild.api.common.Scope;
import ebuild.util.Callable;
import ebuild.util.CollectionUtil;
import ebuild.util.FileUtil;
import ebuild.util.IOUtil;

public class JDTProject {
    final IEBuild ebuild;
    final protected IModuleWorkspaceProject module;
	//private boolean groovy = false;
	
	public JDTProject(IGenConfArgument argument){
	    this.ebuild = argument.getEBuild();
		this.module = argument.getModuleWorkspaceProject();
	}
	
    protected String getJREContainer() {
//      String version = module.getConf().getJavaVersion();
//      String name;
//      if("1.6".equals(version)){
//          name = "JavaSE-"+version;
//      }else{
//          name = "J2SE-"+version;
//      }
        // REMARK keep things simple. We change source compliance settings, but changing this gives class version issues 
        // trying to run the ebuild ant files unfortunately (and it's not easy to avoid).
        return "org.eclipse.jdt.launching.JRE_CONTAINER/org.eclipse.jdt.internal.debug.ui.launcher.StandardVMType/JavaSE-1.6";//+name;
    }
    
    protected String getInputArtifactType(){ return "java"; }
    protected String getSourceArtifactType(){ return getInputArtifactType()+".zip"; }
    protected String getTargetArtifactType(){ return "java.classes.jar"; }
    
    protected List<String> listContainers(){ return CollectionUtil.singletonList(getJREContainer()); }
    protected void contributeToDotProject() throws IOException{
        module.addNature("org.eclipse.jdt.core.javanature");
        module.addBuildCommand("org.eclipse.jdt.core.javabuilder");
    }

	private void generateDotClasspath() throws Exception{
		final List<Map> entries = new ArrayList();
		
		new Callable() {
			private void addEntry(String kind, String path, File sourcepath, boolean export){
				Map e = new HashMap();
				e.put("kind", kind);
				e.put("path", path);
				e.put("export", export+"");
				if(sourcepath!=null)
				    e.put("sourcepath", sourcepath.getAbsolutePath());
                
				entries .add(e);
			}
			private Set<IModuleRelease> norecur = new HashSet();

			private File findSourcesArchive(Collection<File> sources, File jar){
			    String name = FileUtil.removeSuffix(jar.getName())+".zip";
			    for(File f: sources){
			        if(f.getName().equals(name)) return f;
			    }
			    return null;
			}

			
            private void addEntries(IModuleRelease mod) {
                if(norecur.contains(mod)) return;
                else norecur.add(mod);
                
                for(Scope s=Scope.TEST; s!=null; s=s.getParent()){
                    Collection<File> sources = mod.getArtifacts(s,getSourceArtifactType());
	                for(File jar: mod.getArtifacts(s,getTargetArtifactType())){
	                    File sourcesZip = findSourcesArchive(sources, jar);
	                    String path = ebuild.getPathInWorkspace(jar);
                        addEntry("lib", path, sourcesZip, true);
	                    
	                }
	                
	                for(IModuleRelease dep: mod.getDependencies(s)){
	                    addEntries(dep);
	                }
	            }
            }
			private boolean addSrcEntry(String path){
				if(!hasFile(path)) return false; 
				addEntry("src", path, null, true);
				return true;
			}
			private void addSrcEntries(String kind, Scope scope) throws Exception{
				String path = module.getArtifactPath(kind, scope, getInputArtifactType());
				addSrcEntry(path);
//				if(addSrcEntry(path) && FileUtil.hasFileWithSuffix(getFile(path),".groovy")){
//					groovy = true;
//				}
			}
			private void addSrcEntries(Scope scope) throws Exception{
				addSrcEntries("src", scope);
				addSrcEntries("gen", scope);
			}
			public Object run() throws Exception {
				addSrcEntries(Scope.MAIN);
				addSrcEntries(Scope.POKE);
				addSrcEntries(Scope.TEST);
				
				addEntry("output","bin", null, true);
				
				for(String s: listContainers()){
					addEntry("con",s, null, false );
				}
				
				for(IModuleProject mod:  module.getWorkspaceDependencies(Scope.TEST)){
				    addEntry("src","/"+mod.getName(), null, true);
				}
				
				for(IModuleRelease mod: module.getReleasedDependencies(Scope.TEST)){
				    addEntries(mod);
				}
                
                for(Scope s = Scope.TEST; s!=null; s=s.getParent()){
                    Collection<File> checkedArtifacts = module.getCheckedInArtifacts(s,"java.classes.jar");
                    for(File a: checkedArtifacts){
                        addEntry("lib","checkedin/"+a.getName(), null, true);
                    }
                }
                return null;
			}

		}.run();
		// source
		
		Collections.sort(entries, new Comparator<Map>() {
			private String getFilename(String path){ return new File(path).getName(); }
			public int compare(Map o1, Map o2) {
				String s1 = getFilename((String)o1.get("path"));
				String s2 = getFilename((String)o2.get("path"));
				return s1.compareTo(s2);
			}
		});
		
		StringTemplate template = getTemplate("classpath");
		template.setAttribute("entries", entries);
        writeFile(".classpath",template.toString());
	}
    
    private void generateDotSettings() throws Exception {
        File dir = getFile(".settings");
        if(!dir.isDirectory() && !dir.mkdir()) throw new Error("Could not make .settings dir");
        
        StringTemplate template = getTemplate("jdt_prefs");
        template.setAttribute("version", module.getJavaVersion());
        writeFile(".settings/org.eclipse.jdt.core.prefs",template.toString());
    }
	
	public void generate() throws Exception {
	    contributeToDotProject();
        generateDotClasspath();
        generateDotSettings();
	}


    public String toString() { return "EclipseProject: "+ module.getName(); }
	
	//////////////
	// UTILITY
	static private StringTemplateGroup getTemplateGroup(String name) throws IOException{
		URL url = JDTProject.class.getResource(name);
        InputStream is = url.openStream();
        return new StringTemplateGroup(new InputStreamReader(is), DefaultTemplateLexer.class);
    }
	
	static private StringTemplate getTemplate(String name) throws IOException{
		return getTemplateGroup(name+".stg").getInstanceOf(name);
	}
	
    
    public File getFile(String path) {
        return module.getProjectFile(path); 
    }
    public boolean hasFile(String path) {
        return getFile(path).exists(); 
    }
    public void writeFile(String path, String contents) throws IOException {
        IOUtil.stringToFile(contents, module.getProjectFile(path)); 
    }
    
}
