package ebuild.lib.remap;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.Arrays;
import java.util.jar.JarFile;
import java.util.jar.JarOutputStream;

import org.objectweb.asm.ClassReader;
import org.objectweb.asm.ClassWriter;
import org.objectweb.asm.commons.Remapper;
import org.objectweb.asm.commons.RemappingClassAdapter;

import ebuild.util.file.FileContainer;

public class RemappingUtil {
	
	// REMARK - really should create zip/dir abstraction, for working with either
	static public RemapReport remapDir(Remapper remapper, File inDir, File outDir) throws IOException {
		FileContainer in = new FileContainer.Folder(inDir);
		FileContainer out = new FileContainer.Folder(outDir);
		return remap(remapper,in,out);
	}
	
	static public RemapReport remapJar(Remapper remapper, File inFile, File outFile) throws IOException{
		FileContainer in = new FileContainer.ZipRead(new JarFile(inFile));
		FileContainer out = new FileContainer.ZipWrite(new JarOutputStream(new FileOutputStream(outFile)));
		return remap(remapper,in,out);
	}

	static public RemapReport remap(Remapper remapper, FileContainer in, FileContainer out) throws IOException{
		RemapReport report = new RemapReport();
		for(FileContainer.Entry e0: in.list()){
			String name = e0.getName();
			if(!name.endsWith(".class")) continue;
			
			String class0 = e0.getFolderPath()+"/"+e0.getPrefix();
			String class1 = remapper.map(class0);
			// remap all classes, but check ones which change, but are not remapped
			
			byte[] d0 = e0.readBytes();
			byte[] d1 = remapEntry(remapper, e0);
			
			if(class0.equals(class1)){
				byte[] d0_ = nullMap(d0);
				if(!Arrays.equals(d0_,d1)){
					report.shouldHaveBeenMapped.add(class0);
				}else{
					report.skipped.add(class0);
				}
			}else{
				FileContainer.Entry e1 = out.newEntry(class1+".class");
				e1.writeBytes(d1);
				report.remapped.put(class0,class1);
			}
		}
		out.close();
		return report;
	}
	
	static private byte[] nullMap(byte[] d0) {
		try {
			return remapClass(new Remapper() {}, d0);
		} catch (RemappingError e) {
			throw new Error(e);
		}
	}
	
	static public byte[] remapEntry(Remapper remapper, FileContainer.Entry e) throws IOException  {
		try{   
			return remapClass(remapper, e.readBytes());
		} catch (RemappingError re) {
			throw new Error("Unable to remap: "+e.getPath()+"\n"+re.getByteString());
		}
	}
	
	static public byte[] remapClass(Remapper remapper, byte[] bytecode) throws RemappingError  {
        try{
			ClassReader classReader = new ClassReader(bytecode);
	        ClassWriter classWriter = new ClassWriter(classReader, 0);
	        classReader.accept(
	        		new RemappingClassAdapter(classWriter, remapper),
	        		0
	        );
	        return classWriter.toByteArray();
        }catch(Exception e){
        	throw new RemappingError(bytecode);
        }
    }
	
	static private class RemappingError extends Throwable{
		final byte[] bytes;
		RemappingError(byte[] bytes){ this.bytes = bytes; }
		String getByteString(){ return formatByteArr(bytes); }
	}
	
	static private String formatByteArr(byte[] bs){
		StringBuilder sb = new StringBuilder(bs.length*5);
		sb.append('{');
		boolean first = true;
		for(byte b: bs){
			if(first) first = false;
			else sb.append(',');
			sb.append(""+b);
		}
		sb.append('}');
		return sb.toString();
	} 
}
