package ebuild.util;

import static ebuild.util.FileUtil.deleteDirOrFail;
import static ebuild.util.FileUtil.mkdirOrFail;
import static ebuild.util.FileUtil.moveDir;

import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Enumeration;
import java.util.List;
import java.util.jar.JarEntry;
import java.util.jar.JarOutputStream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import java.util.zip.ZipOutputStream;

public class ZipUtil {
    
	

    
    
    //////////////////
    // ZIP
    ///////////
    
    //recursive function to add a direcotry with all sub-directories to zip z
    //output format (in the zip) is linux compatible, i.e. with "/" seperating dirs
    //while the input might be from windows machines, as might happen in mixed
    //condor networks
    static private void zip(File x,String path,ZipOutputStream z, FileFilter filter) throws IOException{
    	if(!filter.accept(x)) return;
    	
    	if(!x.exists())
    		throw new IOException("File not found: "+x);
    	if(!x.isDirectory()){
    		z.putNextEntry(new JarEntry(path+"/"+x.getName()));
    		FileInputStream y=new FileInputStream(x);
    		IOUtil.pipeKeepOpen(y, z);
    		z.closeEntry();
    		y.close();
    	}
    	else  //recurse
    	{
    		if(!"".equals(path))
    			path=path+"/";
    		//z.putNextEntry(new ZipEntry(path));
    		//z.closeEntry();
    		zipDir(x, path+x.getName(), z, filter);
    	}
    }
    
    static private final FileFilter ACCEPT_ALL = new FileFilter(){public boolean accept(File pathname) { return true; }};

    static private void zipDir(File x,String path,ZipOutputStream z, FileFilter filter) throws IOException{
		for(File x2: x.listFiles()){
			zip(x2,path,z,filter);
		}
    }
    static public void zipDirectories(File archive, List<File> dirs) throws IOException{
        zipDirectories(archive, dirs, ACCEPT_ALL);
    }
    static public void zipDirectories(File archive, List<File> dirs, FileFilter filter) throws IOException{
		ZipOutputStream z=new ZipOutputStream(new FileOutputStream(archive));
		for(File dir: dirs){
			zipDir(dir, "", z, filter);
		}
		z.close();
    }
    static public void zipDirectoryIncludeName(File archive, File dir) throws IOException{
    	zipDirectoryIncludeName(archive, dir, ACCEPT_ALL);
    }
    static public void zipDirectoryIncludeName(File archive, File dir, FileFilter filter) throws IOException{
    	zipDirectory(archive, dir, dir.getName(), filter);
    }

    static public void zipDirectory(File archive, File dir, String prefix) throws IOException{
    	zipDirectory(archive, dir, prefix, ACCEPT_ALL);
    }
    static public void zipDirectory(File archive, File dir, String prefix, FileFilter filter) throws IOException{
    	ZipOutputStream z=new ZipOutputStream(new FileOutputStream(archive));
		zipDir(dir, prefix, z, filter);
		z.close();
    }
    static public void jarDirectories(File archive, List<File> dirs) throws IOException{
        jarDirectories(archive, dirs, ACCEPT_ALL);
    }
    static public void jarDirectories(File archive, List<File> dirs, FileFilter filter) throws IOException{
    	JarOutputStream z=new JarOutputStream(new FileOutputStream(archive));
		for(File dir: dirs){
			zipDir(dir, "", z, filter);
		}
		z.close();
    }
    
    
    
    
    
    
    
    //////////////////
    // UNZIP
    ///////////
    static public void unzipArchiveAndRename(File archive, File outputDir) throws IOException {
    	// Create temp file. 
    	File tempDir = FileUtil.createTempDirectory("unzipTo");  
    	unzipArchive(archive, tempDir);
    	
    	String[] dirs = tempDir.list();
    	if(dirs.length!=1) 
    		throw new IOException("Expected single top dir in archive got "+dirs.length+": "+StringUtil.join(",", dirs));
    	
    	File top = new File(tempDir, dirs[0]);
    	moveDir(top, outputDir);
    	deleteDirOrFail(tempDir);
    }
    
    
    static public void unzipArchive(File archive, File outputDir) throws IOException {
        ZipFile zipfile = new ZipFile(archive);
        for (Enumeration<? extends ZipEntry> e = zipfile.entries(); e.hasMoreElements(); ) {
            ZipEntry entry = e.nextElement();
            unzipEntry(zipfile, entry, outputDir);
        }
    }

    static private void unzipEntry(ZipFile zipfile, ZipEntry entry, File outputDir) throws IOException {
        if (entry.isDirectory()) {
        	mkdirOrFail(new File(outputDir, entry.getName()));
        	return;
        }
        
        File outputFile = new File(outputDir, entry.getName());
        mkdirOrFail(outputFile.getParentFile());
        
        InputStream inputStream = zipfile.getInputStream(entry);
        OutputStream outputStream = new FileOutputStream(outputFile);
        try {
            IOUtil.pipe(inputStream, outputStream);
        } finally {
            outputStream.close();
            inputStream.close();
        }
    }
}
