package ebuild.util;

import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.jar.JarEntry;
import java.util.jar.JarOutputStream;
import java.util.zip.ZipOutputStream;

public class FileUtil {
	
	static public Date readTimestampFile(File dir) {
		try {
			File f = new File(dir,"timestamp");
			if(!f.isFile()) return null;
			return java.sql.Timestamp.valueOf(IOUtil.fileToString(new File(dir,"timestamp")));
		} catch (IOException e) {
			throw new Error(e);
		}
	}
	
	static public void writeTimestampFile(File dir, Date timestamp){
		try {
			IOUtil.stringToFile(new java.sql.Timestamp(timestamp.getTime()).toString(),new File(dir,"timestamp"));
		} catch (IOException e) {
			throw new Error(e);
		}
	}

    static public class Traversal<R,E extends Exception> {
        protected boolean exit = false;
        public final void traverse( final File f ) throws E {
            if(exit) return;
            if (f.isDirectory()) {
                if(f.getName().startsWith("."))
                    return;
                onDirectory(f);
                final File[] childs = f.listFiles();
                for( File child : childs ) {
                    traverse(child);
                }
                return;
            }
            onFile(f);
        }

        protected void onDirectory( final File d ) { }

        protected void onFile( final File f ) throws E {}

        public R go(File f) throws E{
            if(f.exists())
                traverse(f);
            return finish();
        }

        public R finish(){
            return null;
        }
    }
    static public boolean fileExists(File dir, String name){
        return new File(dir,name).isFile();
    }
    static public boolean hasFile(File dir, final FileFilter filter) throws Exception{
        return (Boolean) new Traversal(){
            boolean found = false;
            public void onFile(File f) { 
                if(filter.accept(f)) {
                    found = true; 
                    exit = true;
                }
            }
            public Object finish() { return found; }
        }.go(dir);
    }

    static public boolean hasFileWithSuffix(File dir, final String suffix) throws Exception{
        return hasFile(dir,new FileFilter() {
            public boolean accept(File pathname) {
                return pathname.getName().endsWith(suffix);
            }
        });
    }

    static public boolean deleteDir(File dir) {
        if (!dir.exists()) return true;
        if (dir.isDirectory()) {
            String[] children = dir.list();
            for (int i=0; i<children.length; i++) {
                boolean success = deleteDir(new File(dir, children[i]));
                if (!success) {

                    return false;
                }
            }
        }
        // The directory is now empty so delete it
        return dir.delete();
    }

    static public String relativePath(File ancestor, File f) {
        String path = f.getName();
        File p = f;
        while(!(p=p.getParentFile()).equals(ancestor)){
            path=p.getName()+"/"+path; 
        }
        return path;
    }
    static public String fileSuffix(File f){
        return fileSuffix(f.getName());
    }
    static public String fileSuffix(String name){
        int dot = name.indexOf('.');
        if(dot==-1) return null;
        return name.substring(dot+1);
    }

    static public String replaceSuffix(String name, String suffix) {
        return name = removeSuffix(name)+"."+suffix;
    }
    static public String removeSuffix(String name) {
        int dot = name.lastIndexOf('.');
        if(dot==-1) return name;
        return name = name.substring(0,dot);
    }

    static public File copyToDirIfNewer(File f, File dir) throws IOException {
        File r =  new File(dir, f.getName());
        copyIfNewer(f,r);
        return r;
    }

    static public File copyToDir(File f, File dir) throws IOException {
        File r =  new File(dir, f.getName());
        copy(f,r);
        return r;
    }
    
    static public void copyFileIfNewer(File source, File dest) throws IOException {
        if(COPYFILTER_IFNEWER.accept(source,dest)) return;
        doCopyFile(source, dest);
    } 
    static public void copyFile(File source, File dest) throws IOException {
        doCopyFile(source, dest);
    } 

    static public void copyIfNewer(File source, File dest) throws IOException {
        copy(source, dest, COPYFILTER_IFNEWER);
    }
    static public void copy(File source, File dest) throws IOException {
        copy(source, dest, COPYFILTER_ALL);
    }

    static public File createTempDirectory(String name) throws IOException{
    	File r =  File.createTempFile("", "tmp");
    	r.delete();
    	r.mkdirs();
    	return r;
    }
    
    static final public FileFilter FILEFILTER_ALL = new FileFilter(){
    	public boolean accept(File pathname) {
    		return true;
    	}
    };
    
    static final public CopyFilter COPYFILTER_ALL = new CopyFilter(){
        public boolean accept(File source, File dest) {
            return true;
        }
    };
    static final public CopyFilter COPYFILTER_IFNEWER = new CopyFilter(){
        public boolean accept(File source, File dest) {
            if(source.isDirectory()) return true;
            return dest.lastModified()>=source.lastModified();
        }
    };

    static public interface CopyFilter{
        public boolean accept(File source, File dest);
    }
    
    static public CopyFilter newCopyFilter(final FileFilter filter){
    	return new CopyFilter() {
			public boolean accept(File source, File dest) {
				return filter.accept(source);
			}
		};
    }
    
    static public void copy(File source, File dest, FileFilter filter) throws IOException{
    	copy(source, dest, newCopyFilter(filter));
    }
    
    static public void copy(File source, File dest, CopyFilter filter) throws IOException{
        if(!filter.accept(source, dest)) return;
        if (source.isDirectory()){
            if (!dest.exists()){
                dest.mkdirs();
            }

            for(File source2: source.listFiles()) {
                copy(source2, new File(dest, source2.getName()), filter);
            }
        } else{
            if(!source.exists()){
                throw new IOException("File or directory does not exist: "+source);
            }else{
                doCopyFile(source, dest);
            }
        }
    }

    
    static private void doCopyFile(File source, File dest) throws IOException {
        FileChannel in = null, out = null;
        try {     
            dest.getParentFile().mkdirs();
            
            in = new FileInputStream(source).getChannel();
            out = new FileOutputStream(dest).getChannel();

            long size = in.size();
            MappedByteBuffer buf = in.map(FileChannel.MapMode.READ_ONLY, 0, size);
            out.write(buf);
        } finally {
            if (in != null)          in.close();
            if (out != null)     out.close();
        }
    }
    
    static public long deepLastModified(File root) {
        return (Long) new Traversal<Long,RuntimeException>(){
            long lastModified=Long.MIN_VALUE;
            @Override public void onFile(File f) {
                lastModified = Math.max(lastModified,f.lastModified());
            }
            @Override protected void onDirectory(File d) {
            	lastModified = Math.max(lastModified,d.lastModified());
            }
            @Override public Long finish() {
                return lastModified;
            }
        }.go(root);
    }
    
    static public long lastModified(Collection<File> files) {
        long r = 0;
        for(File f: files){
            r = Math.max(r,f.lastModified());
        }
        return r;
 
    }
    static public Collection<String> deepList(File f) {
        Collection<String> r = new ArrayList();
        for(File f2: f.listFiles()){
            deepList_(r, f2.getName(), f2);
        }
        return r;
    }
    static private void deepList_(Collection<String> r, String path, File f){
        if(f.isDirectory()){
            r.add(path+"/");
            for(File f2: f.listFiles()){
                String path2 = path+"/"+f2.getName();
                deepList_(r, path2, f2);
            }
        }else{
            r.add(path);
        }
    }

    static public File canonicalFile(File f){
        try{
            return f.getCanonicalFile();
        }catch(IOException e){
            return f;
        }
    }
    static public void deleteOrFail(File f) throws IOException {
        if(!f.delete()){
            throw new IOException("Unable to delete file: "+f.getAbsolutePath());
        }
    }


    //recursive function to add a direcotry with all sub-directories to zip z
    //output format (in the zip) is linux compatible, i.e. with "/" seperating dirs
    //while the input might be from windows machines, as might happen in mixed
    //condor networks
    static private void zip(File x,String path,ZipOutputStream z, FileFilter filter) throws IOException{
    	if(!filter.accept(x)) return;
    	
    	if(!x.exists())
    		throw new IOException("File not found: "+x);
    	if(!x.isDirectory()){
    		z.putNextEntry(new JarEntry(path+"/"+x.getName()));
    		FileInputStream y=new FileInputStream(x);
    		IOUtil.pipeKeepOpen(y, z);
    		z.closeEntry();
    		y.close();
    	}
    	else  //recurse
    	{
    		if(!path.isEmpty())
    			path=path+"/";
    		//z.putNextEntry(new ZipEntry(path));
    		//z.closeEntry();
    		zipDir(x, path+x.getName(), z, filter);
    	}
    }
    
    static private final FileFilter ACCEPT_ALL = new FileFilter(){public boolean accept(File pathname) { return true; }};

    static private void zipDir(File x,String path,ZipOutputStream z, FileFilter filter) throws IOException{
		for(File x2: x.listFiles()){
			zip(x2,path,z,filter);
		}
    }
    static public void zipDirectories(File archive, List<File> dirs) throws IOException{
        zipDirectories(archive, dirs, ACCEPT_ALL);
    }
    static public void zipDirectories(File archive, List<File> dirs, FileFilter filter) throws IOException{
		ZipOutputStream z=new ZipOutputStream(new FileOutputStream(archive));
		for(File dir: dirs){
			zipDir(dir, "", z, filter);
		}
		z.close();
    }
    static public void zipDirectoryIncludeName(File archive, File dir) throws IOException{
    	zipDirectoryIncludeName(archive, dir, ACCEPT_ALL);
    }
    static public void zipDirectoryIncludeName(File archive, File dir, FileFilter filter) throws IOException{
    	ZipOutputStream z=new ZipOutputStream(new FileOutputStream(archive));
		zipDir(dir, dir.getName(), z, filter);
		z.close();
    }
    static public void jarDirectories(File archive, List<File> dirs) throws IOException{
        jarDirectories(archive, dirs, ACCEPT_ALL);
    }
    static public void jarDirectories(File archive, List<File> dirs, FileFilter filter) throws IOException{
    	JarOutputStream z=new JarOutputStream(new FileOutputStream(archive));
		for(File dir: dirs){
			zipDir(dir, "", z, filter);
		}
		z.close();
    }
}
