package ebuild.util.file;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FilterOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Enumeration;
import java.util.List;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import java.util.zip.ZipOutputStream;

import ebuild.util.FileUtil;
import ebuild.util.IOUtil;

public abstract class FileContainer {
	static final public Filter FILTER_ALL = new Filter(){
		public boolean accept(Entry entry) { return true; }
	};
	
	static public interface Filter{
		public boolean accept(Entry entry); 
	}
	
	static public abstract class Entry{

		abstract public String getPath();
		public String getName(){
			String path = getPath();
			int slash = path.lastIndexOf('/');
			if(slash!=-1) return path.substring(slash+1);
			return path;
		}
		public String getFolderPath(){
			String path = getPath();
			int slash = path.lastIndexOf('/');
			if(slash!=-1) return path.substring(0,slash);
			return null;
		}
		public String getSuffix(){
			String name = getName();
			int dot = name.lastIndexOf('.');
			if(dot!=-1) return name.substring(dot+1);
			return name;
		}
		public String getPrefix(){
			return FileUtil.removeSuffix(getName());
		}
		abstract public InputStream read() throws IOException;
		abstract public OutputStream write() throws IOException;
		public byte[] readBytes() throws IOException{ return IOUtil.inputStreamToBytes(read()); }
		public void writeBytes(byte[] bytes) throws IOException{ 
			OutputStream out = write();
			InputStream in = new ByteArrayInputStream(bytes);
			IOUtil.pipe(in, out);
		}
	}
	
	public Iterable<Entry> list(){
		return list(FILTER_ALL);
	}
	
	public void close() throws IOException {}
	abstract public Iterable<Entry> list(Filter filter);
	abstract public Entry newEntry(String path) throws IOException;
	
	
	
	static public class ZipWrite extends FileContainer{
		final ZipOutputStream zout;
		public ZipWrite(File archiveFile) throws IOException { this(new ZipOutputStream(new FileOutputStream(archiveFile))); }
		public ZipWrite(ZipOutputStream out){ this.zout = out; }
		private Error readError(){ return new Error(getClass().getSimpleName()+" write only!"); }

		public Iterable<FileContainer.Entry> list(Filter filter) { throw readError(); }
		public FileContainer.Entry newEntry(String path) throws IOException {
			ZipEntry e = new ZipEntry(path);
			zout.putNextEntry(e);
			return new Entry(e);
		}
		
		// FIXME really should do checks here, so concurrent creation of writing of entries fails obviously ...
		public class Entry extends FileContainer.Entry{
			final ZipEntry zipEntry;
			public Entry(ZipEntry zipEntry) {
				this.zipEntry = zipEntry;
			}
			public String getPath() { return zipEntry.getName(); }
			public InputStream read() throws IOException { throw readError(); }
			public OutputStream write() throws IOException {
				return new FilterOutputStream(zout){
					public void close() throws IOException {
						flush();
						zout.closeEntry();
					}
				};
			}
		}
		public void close() throws IOException { zout.close(); }
	}

	
	static public class ZipRead extends FileContainer{
		final ZipFile archive;
		public ZipRead(File archiveFile) throws IOException { this(new ZipFile(archiveFile)); }
		public ZipRead(ZipFile archive){ this.archive = archive; }
		private Error writeError(){ return new Error(getClass().getSimpleName()+" read only!"); }

		public Iterable<FileContainer.Entry> list(Filter filter) {
			List<FileContainer.Entry> r = new ArrayList(archive.size());
			Enumeration<? extends ZipEntry> zes = archive.entries();
			while(zes.hasMoreElements()){
				ZipEntry ze = zes.nextElement();
				Entry e = new Entry(ze);
				if(filter.accept(e))
					r.add(e);
			}
			return r;
		}
		public FileContainer.Entry newEntry(String path) { throw writeError(); }
	
		public class Entry extends FileContainer.Entry{
			final ZipEntry zipEntry;
			public Entry(ZipEntry zipEntry) {
				this.zipEntry = zipEntry;
			}
			public String getPath() { return zipEntry.getName(); }
			public InputStream read() throws IOException {
				return archive.getInputStream(zipEntry);
			}
			public OutputStream write() throws IOException { throw writeError(); }
		}
	}

	static public class Folder extends FileContainer{
		final File top;
		public Folder(File top){ this.top = top; }
		
		public Iterable<FileContainer.Entry> list(Filter filter) {
			Collection<String> ps = FileUtil.deepList(top);
			List<FileContainer.Entry> r = new ArrayList(ps.size());
			for(String p: ps){
				File f = new File(top, p);
				Entry e = new Entry(f, p);
				if(filter.accept(e)){
					r.add(e);
				}
			}
			return r;
		}
		public FileContainer.Entry newEntry(String path) {
			return new Entry(new File(top, path), path);
		}

		public class Entry extends FileContainer.Entry{
			final File file;
			final String path;
			Entry(File file, String path){
				this.file = file;
				this.path = path;
			}
			public String getPath() { return path; }
			public InputStream read() throws IOException { return new FileInputStream(file); }
			public OutputStream write() throws IOException {
				FileUtil.mkdirOrFail(file.getParentFile());
				return new FileOutputStream(file);
			}
		}
	}

}
