package ebuild.util;

import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.List;

public class FileUtil {
	
	static public Date readTimestampFile(File dir) {
		try {
			File f = new File(dir,"timestamp");
			if(!f.isFile()) return null;
			return java.sql.Timestamp.valueOf(IOUtil.fileToString(new File(dir,"timestamp")));
		} catch (IOException e) {
			throw new Error(e);
		}
	}
	
	static public void writeTimestampFile(File dir, Date timestamp){
		try {
			IOUtil.stringToFile(new java.sql.Timestamp(timestamp.getTime()).toString(),new File(dir,"timestamp"));
		} catch (IOException e) {
			throw new Error(e);
		}
	}

    static public class Traversal<R,E extends Exception> {
        protected boolean exit = false;
        public final void traverse( final File f ) throws E {
            if(exit) return;
            if (f.isDirectory()) {
                if(f.getName().startsWith("."))
                    return;
                boolean cont = onDirectory(f);
                if(cont){
	                final File[] childs = f.listFiles();
	                for( File child : childs ) {
	                    traverse(child);
	                }
	            }
            }else{
            	onFile(f);
            }
        }

        protected boolean onDirectory( final File d ) throws E { return true; }

        protected void onFile( final File f ) throws E {}

        public R go(File f) throws E{
            if(f.exists())
                traverse(f);
            return finish();
        }

        public R finish(){
            return null;
        }
    }
    static public boolean fileExists(File dir, String name){
        return new File(dir,name).isFile();
    }
    static public boolean hasFile(File dir, final FileFilter filter) throws Exception{
        return (Boolean) new Traversal(){
            boolean found = false;
            public void onFile(File f) { 
                if(filter.accept(f)) {
                    found = true; 
                    exit = true;
                }
            }
            public Object finish() { return found; }
        }.go(dir);
    }

    static public boolean hasFileWithSuffix(File dir, final String suffix) throws Exception{
        return hasFile(dir,new FileFilter() {
            public boolean accept(File pathname) {
                return pathname.getName().endsWith(suffix);
            }
        });
    }

    static public boolean deleteDir(File dir) {
        if (!dir.exists()) return true;
        if (dir.isDirectory()) {
            String[] children = dir.list();
            for (int i=0; i<children.length; i++) {
                boolean success = deleteDir(new File(dir, children[i]));
                if (!success) {

                    return false;
                }
            }
        }
        // The directory is now empty so delete it
        return dir.delete();
    }

    static public String relativePath2Parent(File child, File parent) {
    	if(parent.equals(child)) return ".";
    	
    	String path = "..";
        File p = child;
        while(!(p=p.getParentFile()).equals(parent)){
            path=".."+"/"+path; 
        }
        return path;
    }
    
    static public String relativePath2Child(File parent, File child) {
        if(parent.equals(child)) return ".";
    	
    	String path = child.getName();
        File p = child;
        while(!(p=p.getParentFile()).equals(parent)){
            path=p.getName()+"/"+path; 
        }
        return path;
    }
    
    @Deprecated
    static public String relativePath(File ancestor, File f) {
    	return relativePath2Child(ancestor, f);
    }
    static public String fileLastSuffix(File f){
        return fileLastSuffix(f.getName());
    }
    static public String fileLastSuffix(String name){
        int dot = name.lastIndexOf('.');
        if(dot==-1) return null;
        return name.substring(dot+1);
    }
    
    static public String fileSuffix(File f){
        return fileSuffix(f.getName());
    }
    static public String fileSuffix(String name){
        int dot = name.indexOf('.');
        if(dot==-1) return null;
        return name.substring(dot+1);
    }

    static public String replaceSuffix(String name, String suffix) {
        return name = removeSuffix(name)+"."+suffix;
    }
    static public String removeSuffix(String name) {
        int dot = name.lastIndexOf('.');
        if(dot==-1) return name;
        return name = name.substring(0,dot);
    }

    static public File copyToDirIfNewer(File f, File dir) throws IOException {
        File r =  new File(dir, f.getName());
        copyIfNewer(f,r);
        return r;
    }

    static public File copyToDir(File f, File dir) throws IOException {
        File r =  new File(dir, f.getName());
        copy(f,r);
        return r;
    }
    
    static public void copyFileIfNewer(File source, File dest) throws IOException {
        if(COPYFILTER_IFNEWER.accept(source,dest)) return;
        doCopyFile(source, dest);
    } 
    static public void copyFile(File source, File dest) throws IOException {
        doCopyFile(source, dest);
    } 

    static public void copyIfNewer(File source, File dest) throws IOException {
        copy(source, dest, COPYFILTER_IFNEWER);
    }
    static public void copy(File source, File dest) throws IOException {
        copy(source, dest, COPYFILTER_ALL);
    }
    static public File createTempDirectory() throws IOException{ return createTempDirectory("directory"); }
    static public File createTempDirectory(String name) throws IOException{
    	File r =  File.createTempFile(name, null);
    	r.delete();
    	r.mkdirs();
    	return r;
    }
    
    static public File createTempFile() throws IOException{ return createTempFile("file"); }
    static public File createTempFile(String name) throws IOException{
    	return File.createTempFile(name, null);
    }
    
    static final public FileFilter FILEFILTER_ALL = new FileFilter(){
    	public boolean accept(File pathname) {
    		return true;
    	}
    };
    
    static final public CopyFilter COPYFILTER_ALL = new CopyFilter(){
        public boolean accept(File source, File dest) {
            return true;
        }
    };
    static final public CopyFilter COPYFILTER_IFNEWER = new CopyFilter(){
        public boolean accept(File source, File dest) {
            if(source.isDirectory()) return true;
            return dest.lastModified()>=source.lastModified();
        }
    };

    static public interface CopyFilter{
        public boolean accept(File source, File dest);
    }
    
    static public CopyFilter newCopyFilter(final FileFilter filter){
    	return new CopyFilter() {
			public boolean accept(File source, File dest) {
				return filter.accept(source);
			}
		};
    }
    
    static public void copy(File source, File dest, FileFilter filter) throws IOException{
    	copy(source, dest, newCopyFilter(filter));
    }
    
    static public void copy(File source, File dest, CopyFilter filter) throws IOException{
        if(!filter.accept(source, dest)) return;
        if (source.isDirectory()){
            if (!dest.exists()){
                dest.mkdirs();
            }

            for(File source2: source.listFiles()) {
                copy(source2, new File(dest, source2.getName()), filter);
            }
        } else{
            if(!source.exists()){
                throw new IOException("File or directory does not exist: "+source);
            }else{
                doCopyFile(source, dest);
            }
        }
    }

    
    static private void doCopyFile(File source, File dest) throws IOException {
        FileChannel in = null, out = null;
        try {     
            dest.getParentFile().mkdirs();
            
            in = new FileInputStream(source).getChannel();
            out = new FileOutputStream(dest).getChannel();

            long size = in.size();
            MappedByteBuffer buf = in.map(FileChannel.MapMode.READ_ONLY, 0, size);
            out.write(buf);
        } finally {
            if (in != null)          in.close();
            if (out != null)     out.close();
        }
    }
    
    static public long deepLastModified(File root) {
        return (Long) new Traversal<Long,RuntimeException>(){
            long lastModified=Long.MIN_VALUE;
            @Override public void onFile(File f) {
                lastModified = Math.max(lastModified,f.lastModified());
            }
            @Override protected boolean onDirectory(File d) {
            	lastModified = Math.max(lastModified,d.lastModified());
            	return true;
            }
            @Override public Long finish() {
                return lastModified;
            }
        }.go(root);
    }
    
    static public long lastModified(Collection<File> files) {
        long r = 0;
        for(File f: files){
            r = Math.max(r,f.lastModified());
        }
        return r;
 
    }
    static public Collection<String> deepList(File f) {
        Collection<String> r = new ArrayList();
        for(File f2: f.listFiles()){
            deepList_(r, f2.getName(), f2);
        }
        return r;
    }
    static private void deepList_(Collection<String> r, String path, File f){
        if(f.isDirectory()){
            r.add(path+"/");
            for(File f2: f.listFiles()){
                String path2 = path+"/"+f2.getName();
                deepList_(r, path2, f2);
            }
        }else{
            r.add(path);
        }
    }

    static public File canonicalFile(File f){
        try{
        	if(f==null) return null;
            return f.getCanonicalFile();
        }catch(IOException e){
            return f;
        }
    }
    static public void deleteOrFail(File f) throws IOException {
        if(!f.delete())
            throw new IOException("Unable to delete file: "+f.getAbsolutePath());
    }
    static public void deleteDirOrFail(File d) throws IOException {
    	if(!deleteDir(d))
            throw new IOException("Unable to delete dir: "+d.getAbsolutePath());
    }    
    
    static public void mkdirOrFail(File d) throws IOException{
    	if(!d.isDirectory() && !d.mkdirs())
            throw new IOException("Unable to create directories: "+d.getAbsolutePath());
    }
    
    static public void moveOrFail(File source, File dest) throws IOException{
    	if(!source.renameTo(dest)){
            throw new IOException("Unable to move file from '"+source+"' to '"+dest+"'");
    	}
    }
    
    static public void deleteOrError(File f) {
        if(!f.delete())
            throw new Error("Unable to delete file: "+f.getAbsolutePath());
    }
    static public void deleteDirOrError(File d) {
    	if(!deleteDir(d))
            throw new Error("Unable to delete dir: "+d.getAbsolutePath());
    }    
    
    static public void mkdirOrError(File d) {
    	if(!d.isDirectory() && !d.mkdirs())
            throw new Error("Unable to create directories: "+d.getAbsolutePath());
    }
    
    
    static public void mergeMove(File sourceDir, File destDir) throws IOException{
    	merge(sourceDir, destDir, true);
    }
    
    static public void mergeCopy(File sourceDir, File destDir) throws IOException{
    	merge(sourceDir, destDir, false);
    }
    
    static private void merge(File source, File dest, boolean move) throws IOException{
        if(source.isDirectory()){
        	mkdirOrFail(dest);
    		for(File s: source.listFiles()){
    			File d = new File(dest, s.getName());
    			merge(s, d, move);
    		}
        }else{
        	if(dest.exists()) deleteDirOrFail(dest);
        	if(move) moveOrFail(source, dest);
        	else copyFile(source, dest);
        }
    }
    
    static public void moveDir(File sourceParent, File destParent) throws IOException{
        mkdirOrFail(destParent);
	
		for(File source: sourceParent.listFiles()){
			File dest = new File(destParent, source.getName());
			moveOrFail(source, dest);
		}
    }
	
    @Deprecated static public void zipDirectories(File archive, List<File> dirs) throws IOException{ ZipUtil.zipDirectories(archive, dirs); }
    @Deprecated static public void zipDirectories(File archive, List<File> dirs, FileFilter filter) throws IOException{ ZipUtil.zipDirectories(archive, dirs, filter); }
    @Deprecated static public void zipDirectoryIncludeName(File archive, File dir) throws IOException{ ZipUtil.zipDirectoryIncludeName(archive, dir); }
    @Deprecated static public void zipDirectoryIncludeName(File archive, File dir, FileFilter filter) throws IOException{ ZipUtil.zipDirectoryIncludeName(archive, dir, filter); }
    @Deprecated static public void zipDirectory(File archive, File dir, String prefix) throws IOException{ ZipUtil.zipDirectory(archive, dir, prefix); }
    @Deprecated static public void zipDirectory(File archive, File dir, String prefix, FileFilter filter) throws IOException{ ZipUtil.zipDirectory(archive, dir, prefix, filter); }
    @Deprecated static public void jarDirectories(File archive, List<File> dirs) throws IOException{ ZipUtil.jarDirectories(archive, dirs); }
    @Deprecated static public void jarDirectories(File archive, List<File> dirs, FileFilter filter) throws IOException{ ZipUtil.jarDirectories(archive, dirs, filter); }
}
