package ebuild.util;

import java.io.IOException;
import java.io.OutputStream;
import java.io.PrintStream;
import java.io.PrintWriter;
import java.io.StringWriter;

import ebuild.api.log.ILogger;
import ebuild.api.log.ISwitch;

public class Logger implements ILogger, Appendable{
    static private PrintStream newInterceptStream(final char m){
        return new PrintStream(new OutputStream(){
            private boolean nl = true;
            public void write(int b) throws IOException {
                if(nl){
                    current.out.print(current.indent+m+" ");
                    nl = false;
                }
                if('\n'==b) nl = true;
                current.out.print((char)b);
            }
        });
    }
    

    static private ISwitch ALWAYS_ON = new ISwitch(){
        public boolean isOn() { return true; };
    };
    
    static final protected PrintStream system_err = System.err;
//    static private PrintStream system_out = System.out;
    static private void interceptStd(){ 
        System.setErr(newInterceptStream('!'));
        System.setOut(newInterceptStream('~'));
    }
//    static private void restoreStd(){ 
//        System.setErr(system_err);
//        System.setOut(system_out);
//    }
    
    static{
        interceptStd();
    }
    
    static public Logger newRootLogger(){
        return newRootLogger(system_err);
    }
    static public Logger newRootLogger(PrintStream out){
        return new Logger(out);
    }
    
    
    static private String SEPARATOR = "------------------------------------------------------------------------------------------------------------------------";
    static Logger current;
    
    final ISwitch onSwitch;
    final Logger parent;
    final public PrintStream out;
    final String name;
    final String indent;
    final String prefix;
    final Object identity;
    private String provisionalHeader;
    
    protected Logger(PrintStream out){
        this.onSwitch = ALWAYS_ON;
        this.parent = null;
        this.name = null;
        this.indent = "";
        this.prefix = " ";
        this.out = out;
        this.identity = this;
        current = this;
    }
    
    protected Logger(Logger parent, String name, ISwitch onSwitch, String indent, String prefix, Object identity){
        this.onSwitch = onSwitch;
        this.parent = parent;
        this.name = name;
        this.out = parent.out;
        this.indent = indent;
        this.prefix = prefix;
        this.identity = identity==null?this:identity;
    }
    
    public boolean isOn(){ 
        if(parent!=null && !parent.isOn()) return false;
        return onSwitch.isOn(); 
    }
    //private boolean isRoot(){ return parent==null; }
    private boolean isAncestorOf(Logger l){ 
        if(l==null) return false;
        if(identity.equals(l.identity)) return true;
        return isAncestorOf(l.parent); 
    }    
    
    private boolean isSameIndentation(Logger l){
        return identity.equals(l.identity);
    }
    
    private String padSeparator(String s){
    	return s + SEPARATOR.substring(Math.min(SEPARATOR.length(), s.length()));
// Keeping as it can cause a break if recursing many times
//        return s + SEPARATOR.substring(s.length());
    }
    public Logger newSubLogger(String name) {
        return newSubLogger(name, null);
    }
    public Logger newSubLogger(ISwitch onSwitch) {
        return newSubLogger(null, onSwitch);
    }
    public Logger newSubLogger(String name, ISwitch onSwitch) {
        return newSubLogger(name, onSwitch, "| ");
    }
    
    protected Logger newSubLoggerInstance(String name, ISwitch onSwitch, String indent2, String prefix2, Object identity2){
        return new Logger(this, name, onSwitch, indent2, prefix2, identity2);
    }
    
    public Logger newSubLogger(String name, ISwitch onSwitch, String prefix2) {
        if(onSwitch==null) onSwitch = ALWAYS_ON;
        
        
        String indent2;
        Object identity2;						
        if(name==null){
            indent2 = this.indent;
            identity2 = this.identity;
        }else{
            indent2 = "";
            Logger p = this;
            while(p!=null){
                indent2 += "    ";
                p = p.parent;
            }
            identity2 = null;
        }
        
        prefix2 = indent2+prefix2;
        return newSubLoggerInstance(name, onSwitch, indent2, prefix2, identity2);
    }

    
    public void prime() {
        if(!isSameIndentation(current)){
            if(this.isAncestorOf(current)){
                out.println(padSeparator(indent));
            }else{
                if(parent!=null && parent!=current && parent!=current.parent){
                    parent.log("");
                }
                int insetLength = indent.length()-current.indent.length();
                String inset = SEPARATOR.substring(0, Math.max(0,insetLength));
                out.println(padSeparator(current.indent+inset+"-------- "+name+" "));
            }
            current = this;
        }        
    }
    
    private void doLog(String msg){
        if(msg==null){
            out.print(prefix);
            out.println("*null*");
        }else{
            String[] lines = msg.split("\n");
            for(String l: lines){
                out.print(prefix);
                out.println(l);
            }
        }
//        out.flush();
    }
    
	
	static public class Warning{
		final public String message;
		final public Throwable throwable;
		Warning(String message, Throwable throwable){
			this.message = message;
			this.throwable = throwable;
		}
	}
    
    protected void doLogWarning(Warning w){
    	prime();
        printProvisionalHeading();
        doLog("[WARNING] "+w.message);
        if(w.throwable!=null){
        	log(w.throwable);
        }
    }
    
    public void warn(String msg, Exception e){ doLogWarning(new Warning(msg, e)); }
    public void warn(String msg){ doLogWarning(new Warning(msg, null)); }
    
    
    public void log(String msg){
        if(isOn()){
        	prime();
            printProvisionalHeading();
            doLog(msg);
        }
    }
    
    public void log(Throwable t){
        StringWriter sw = new StringWriter();
        t.printStackTrace(new PrintWriter(sw));
        log(sw.toString());
    }
    

    public void logPartial(String msg) {
        if(isOn()){
            printProvisionalHeading();
            out.print(msg);
            out.flush();
        }
    }

    private String getProvisionalHeading(){
        String r = null;
    	if(parent!=null) r = parent.getProvisionalHeading();
    	if(r==null) r = provisionalHeader;
    	return r;
    }
    
    private void printProvisionalHeading(){
        String h;
    	while((h=getProvisionalHeading())!=null){
            doLog(h);
            unsetProvisionalHeading(h);
        }
    }
    
    public void setProvisionalHeading(String provisionalHeader) {
        this.provisionalHeader = provisionalHeader;
    }
    private void unsetProvisionalHeading(String provisionalHeader) {
        if(this.provisionalHeader==provisionalHeader){
        	this.provisionalHeader = null;
        }
        if(parent!=null) parent.unsetProvisionalHeading(provisionalHeader);
    }
    
    public boolean equals(Object obj) {
        if(this==obj) return true;
        if(obj==null) return false;
        Logger b = (Logger)obj;
        if(!ObjectUtil.areEqual(parent, b.parent)) return false;
        return ObjectUtil.areEqual(name, b.name);
    }
    
	public Appendable append(CharSequence csq) { return append(csq, 0, csq.length()); }
	public Appendable append(CharSequence csq, int start, int end) {
        if(isOn()){
            printProvisionalHeading();
            out.append(csq, start, end);
            out.flush();
        }
        return this;
	}
	public Appendable append(char c) {
		if(isOn()){
            printProvisionalHeading();
            out.append(c);
            out.flush();
        }
        return this;
	}
}
